package ru.yandex.solomon.name.resolver.logbroker;

import java.time.Instant;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.persqueue.PersqueueClient;
import ru.yandex.persqueue.read.ReadSession;
import ru.yandex.persqueue.read.settings.EventHandlersSettings;
import ru.yandex.persqueue.read.settings.ReadSessionSettings;
import ru.yandex.solomon.config.DataSizeConverter;
import ru.yandex.solomon.config.protobuf.TLogbrokerReaderConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.name.resolver.IssueTracker;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.sink.ResourceUpdater;

/**
 * @author Vladimir Gordiychuk
 */
public class PersqueueReader implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(PersqueueReader.class);

    private final String instance;
    private final ResourceUpdater updater;
    private final IssueTracker issueTracker;
    private final Predicate<Resource> filter;
    private final MetricRegistry registry;
    private final PersqueueClient client;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final TLogbrokerReaderConfig config;

    private final AtomicLong generation = new AtomicLong();
    private volatile ReadSession readSession;
    private volatile boolean closed;

    // for manager ui
    Throwable lastError;
    Instant lastErrorTime = Instant.EPOCH;

    public PersqueueReader(
            String instance,
            PersqueueClient client,
            IssueTracker issueTracker,
            ThreadPoolProvider threads,
            ResourceUpdater updater,
            Predicate<Resource> filter,
            TLogbrokerReaderConfig config,
            MetricRegistry registry)
    {
        this.instance = instance;
        this.issueTracker = issueTracker;
        this.updater = updater;
        this.filter = filter;
        this.client = client;
        this.registry = registry.subRegistry("instance", instance);
        this.executor = threads.getExecutorService(
                config.getThreadPoolName(),
                "LogbrokerReader.ThreadPoolName");
        this.timer = threads.getSchedulerExecutorService();
        this.config = config;
    }

    private ReadSessionSettings makeSettings(PersqueueSubscriber subscriber) {
        int memoryLimit = DataSizeConverter.toBytesInt(config.getMaxMemoryUse());
        if (memoryLimit == 0) {
            memoryLimit = 100 << 10; // 100 MiB
        }

        var settings = ReadSessionSettings.newBuilder()
                .consumerName(config.getConsumer())
                .maxMemoryUsage(memoryLimit)
                .executor(executor)
                .eventHandlers(EventHandlersSettings.newBuilder()
                        .commonHandler(subscriber)
                        .executor(executor)
                        .build());

        for (var topic : config.getTopicsList()) {
            settings.addTopic(topic);
        }

        return settings.build();
    }

    public void start() {
        if (readSession != null) {
            throw new IllegalStateException("Already started");
        }

        start(generation.incrementAndGet());

    }

    private void onError(Throwable e, long gen) {
        long nextGen = gen + 1;
        if (this.generation.compareAndSet(gen, nextGen)) {
            long delay = ThreadLocalRandom.current().nextLong(60_000);
            logger.error("persqueue error for instance {}, gen {}, retry after {} ms", instance, gen, delay, e);
            lastError = e;
            lastErrorTime = Instant.now();
            timer.schedule(() -> start(nextGen), delay, TimeUnit.MILLISECONDS);
        }
    }

    private void start(long generation) {
        var subscriber = new PersqueueSubscriber(updater, issueTracker, filter, 1000, e -> onError(e, generation), registry);
        var session = client.createReadSession(makeSettings(subscriber));
        this.readSession = session;
        if (closed) {
            session.close();
        } else {
            session.start();
        }
    }

    @Override
    public void close() {
        closed = true;
        var copy = readSession;
        if (copy != null) {
            copy.close();
        }
    }
}
