package ru.yandex.solomon.name.resolver.logbroker;

import java.io.IOException;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Predicate;

import com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.persqueue.codec.Decode;
import ru.yandex.persqueue.read.PartitionStream;
import ru.yandex.persqueue.read.event.AbstractEventHandler;
import ru.yandex.persqueue.read.event.Event;
import ru.yandex.persqueue.read.event.Message;
import ru.yandex.persqueue.read.event.PartitionStreamClosedEvent;
import ru.yandex.persqueue.read.event.PartitionStreamCreateEvent;
import ru.yandex.persqueue.read.event.PartitionStreamDestroyEvent;
import ru.yandex.solomon.name.resolver.IssueTracker;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.index.ResourceInternerImpl;
import ru.yandex.solomon.name.resolver.sink.ResourceUpdateRequest;
import ru.yandex.solomon.name.resolver.sink.ResourceUpdater;

/**
 * @author Vladimir Gordiychuk
 */
public class PersqueueSubscriber extends AbstractEventHandler implements Flow.Subscriber<Event> {
    private static final Logger logger = LoggerFactory.getLogger(PersqueueSubscriber.class);
    private final ResourceUpdater resourceUpdater;
    private final IssueTracker issueTracker;
    private final Predicate<Resource> filter;
    private final PersqueueSubscriberMetrics metrics;
    private final int maxUncommitted;
    private final Consumer<Throwable> onError;
    private Flow.Subscription subscription;

    public PersqueueSubscriber(ResourceUpdater resourceUpdater, IssueTracker issueTracker, Predicate<Resource> filter, int maxUncommitted, Consumer<Throwable> onError, MetricRegistry registry) {
        this.resourceUpdater = resourceUpdater;
        this.issueTracker = issueTracker;
        this.filter = filter;
        this.maxUncommitted = maxUncommitted;
        this.onError = onError;
        this.metrics = new PersqueueSubscriberMetrics(registry);
        this.metrics.demand.set(0);
        this.metrics.uncommittedLimit.set(maxUncommitted);
    }

    @Override
    public void onSubscribe(Flow.Subscription subscription) {
        this.subscription = subscription;
        requestMore(maxUncommitted);
    }

    @Override
    public void onNext(Event item) {
        metrics.demand.add(-1);
        item.visit(this);
        requestMore(1);
    }

    @Override
    public void onMessage(Message event) {
        var metrics = this.metrics.receiveMessage(event);
        var deferredCommit = new DeferredCommit(event.getOffset(), event.getPartitionStream(), metrics);
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("onMessage {}, {}", event.getPartitionStream().getKey(), silentDecode(event));
            }

            sendUpdates(event, deferredCommit, metrics);
            deferredCommit.commit();
        } catch (Throwable e) {
            logger.error("Unable to process event={}, message={}", event, silentDecode(event), e);
            // TODO: save it to another place?
            deferredCommit.failed();
        }
    }

    @Override
    public void onPartitionStreamClosed(PartitionStreamClosedEvent event) {
        metrics.unassign(event.getPartitionStream());
    }

    @Override
    public void onPartitionStreamCreate(PartitionStreamCreateEvent event) {
        metrics.assign(event.getPartitionStream());
        event.confirm();
    }

    @Override
    public void onPartitionStreamDestroy(PartitionStreamDestroyEvent event) {
        metrics.unassign(event.getPartitionStream());
        event.confirm();
    }

    @Override
    public void onError(Throwable throwable) {
        metrics.unassignAll();
        metrics.errors.inc();
        onError.accept(throwable);
    }

    @Override
    public void onComplete() {
        metrics.unassignAll();
    }

    private void requestMore(int count) {
        long uncommitted = metrics.demand.get() + metrics.uncommittedMessage();
        if (uncommitted + count <= maxUncommitted) {
            metrics.demand.add(count);
            subscription.request(count);
        }
    }

    private void sendUpdates(Message event, DeferredCommit deferredCommit, PartitionStreamMetrics.Metrics metrics) throws IOException {
        try (var input = Decode.decodeStream(event.getCodec(), event.getData());
             var parser = new ResourceParser(input))
        {
            while (parser.hasNext()) {
                var resource = parser.next();
                logger.info("receive: {}, from {}, seqNo {}, offset {}", resource, event.getPartitionStream(), event.getSeqNo(), event.getOffset());
                deferredCommit.addUncommitted();
                var issue = ResourceValidator.STRICT_VALIDATOR.validate(resource);
                if (issue != null) {
                    logger.warn("invalid: {}, from {}, seqNo {}, issue: {}", resource, event.getPartitionStream(), event.getSeqNo(), issue);
                    deferredCommit.failed();
                    metrics.invalidResource();
                    issueTracker.invalidResource(resource, issue);
                    continue;
                }

                if (!filter.test(resource)) {
                    logger.info("filter: {}, from {}, seqNo {}", resource, event.getPartitionStream(), event.getSeqNo());
                    deferredCommit.commit();
                    metrics.filteredResource();
                    continue;
                }

                metrics.acceptedResource();
                resourceUpdater.update(new UpdateRequestImpl(intern(resource), deferredCommit));
            }
        }
    }

    private static Resource intern(Resource resource) {
        return ResourceInternerImpl.INSTANCE.intern(resource);
    }

    private String silentDecode(Message message) {
        try {
            ByteString decoded = Decode.decodeToByteString(message.getCodec(), message.getData());
            return "{codec: RAW, data: " + decoded.toStringUtf8() + "}";
        } catch (Throwable e) {
            return "{codec: " + message.getCodec().name() + ", data: " + message.getData() + "}";
        }
    }

    private final class UpdateRequestImpl implements ResourceUpdateRequest {
        private final Resource resource;
        private final DeferredCommit deferredCommit;
        private final AtomicBoolean committed = new AtomicBoolean();

        public UpdateRequestImpl(Resource resource, DeferredCommit deferredCommit) {
            this.resource = resource;
            this.deferredCommit = deferredCommit;
        }

        @Override
        public Resource resource() {
            return resource;
        }

        @Override
        public void complete() {
            if (!committed.compareAndSet(false, true)) {
                throw new IllegalStateException("Resource " + resource + " already committed");
            }

            deferredCommit.commit();
        }
    }

    private class DeferredCommit {
        private final long offset;
        private final PartitionStream partitionStream;
        private final PartitionStreamMetrics.Metrics metrics;
        private final AtomicInteger uncommitted = new AtomicInteger(1);
        private volatile boolean failed;

        public DeferredCommit(long offset, PartitionStream partitionStream, PartitionStreamMetrics.Metrics metrics) {
            this.offset = offset;
            this.partitionStream = partitionStream;
            this.metrics = metrics;
        }

        public void addUncommitted() {
            int count = uncommitted.getAndIncrement();
            if (count == 0) {
                throw new IllegalStateException(partitionStream + " by offset " + offset + " already committed");
            }
        }

        public void failed() {
            failed = true;
            commit();
        }

        public void commit() {
            if (uncommitted.decrementAndGet() == 0) {
                partitionStream.commit(offset);
                if (failed) {
                    metrics.failed();
                } else {
                    metrics.commit();
                }
                requestMore(1);
            }
        }
    }
}
