package ru.yandex.solomon.name.resolver.logbroker;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.solomon.core.db.model.ReferenceConf;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.name.resolver.ServiceProviderListener;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.ResourceLabels;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceFilter implements Predicate<Resource>, ServiceProviderListener {
    private final boolean enable;

    public ResourceFilter(boolean enable) {
        this.enable = enable;
    }

    @Nullable
    private volatile State snapshot;

    public boolean isReady() {
        return snapshot != null;
    }

    /**
     * @return fail if resource can not be accepted
     */
    @Override
    public boolean test(Resource resource) {
        var copy = snapshot;
        if (copy == null) {
            throw new IllegalStateException("Resource filter not initialized yet");
        }

        if (!enable) {
            return true;
        }

        return copy.test(Key.of(resource));
    }

    @Override
    public void onUpdateServiceProviders(Map<String, ServiceProvider> serviceProviderById) {
        var conditions = serviceProviderById.values()
                .stream()
                .flatMap(provider -> provider.getReferences().stream())
                .map(Cond::of)
                .collect(Collectors.toSet());
        this.snapshot = new State(conditions);
    }

    private static Selector toSelector(String key, List<String> values) {
        if (values.isEmpty()) {
            return Selector.any(key);
        }

        return Selector.glob(key, values.stream().distinct().collect(Collectors.joining("|")));
    }

    private static record Cond(Selector service, Selector type) implements Predicate<Key> {
        public static Cond of(ReferenceConf conf) {
            return new Cond(toSelector(ResourceLabels.SERVICE, conf.services), toSelector(ResourceLabels.TYPE, conf.types));
        }

        @Override
        public boolean test(Key key) {
            return service.match(key.service()) && type.match(key.type);
        }
    }

    private static record Key(String service, String type) {
        public static Key of(Resource resource) {
            return new Key(resource.service, resource.type);
        }
    }

    private static class State implements Predicate<Key> {
        private final Set<Cond> conditions;
        private final Map<Key, Boolean> cache = new ConcurrentHashMap<>();

        public State(Set<Cond> conditions) {
            this.conditions = conditions;
        }

        @Override
        public boolean test(Key key) {
            var cached = cache.get(key);
            if (cached != null) {
                return cached;
            }

            boolean result = false;
            for (var cond : conditions) {
                if (cond.test(key)) {
                    result = true;
                    break;
                }
            }

            cache.put(key, result);
            return result;
        }
    }
}
