package ru.yandex.solomon.name.resolver.logbroker;

import java.io.IOException;
import java.io.InputStream;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.NoSuchElementException;

import javax.annotation.Nonnull;
import javax.annotation.WillClose;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonParser.Feature;
import com.fasterxml.jackson.core.JsonToken;
import com.google.common.base.Strings;
import com.google.common.collect.Iterators;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceParser implements Iterator<Resource>, AutoCloseable {
    private static final JsonFactory factory = new JsonFactory();

    private final JsonParser parser;

    public ResourceParser(String json) throws IOException {
        this.parser = factory.createParser(json);
    }

    public ResourceParser(@WillClose InputStream io) throws IOException {
        this.parser = factory.createParser(io).enable(Feature.AUTO_CLOSE_SOURCE);
    }

    @Nonnull
    public static Resource parse(String json) throws IOException {
        try (var parser = new ResourceParser(json)) {
            return Iterators.getOnlyElement(parser);
        }
    }

    @Nonnull
    public static Resource parse(@WillClose InputStream io) throws IOException {
        try (var parser = new ResourceParser(io)) {
            return Iterators.getOnlyElement(parser);
        }
    }

    private static String nextString(JsonParser parser) throws IOException {
        var token = parser.nextToken();
        switch (token) {
            case VALUE_NULL:
                return "";
            case VALUE_STRING:
                return parser.getText();
            default:
                throw new RuntimeException("expected one of: " + EnumSet.of(JsonToken.VALUE_NULL, JsonToken.VALUE_STRING) + ", got: " + token + ", at " + parser.getCurrentLocation());
        }
    }

    private static long nextTimestamp(JsonParser parser) throws IOException {
        var token = parser.nextToken();
        switch (token) {
            case VALUE_NULL:
                return 0;
            case VALUE_STRING:
                String value = parser.getText();
                if (Strings.isNullOrEmpty(value)) {
                    return 0;
                }

                return InstantUtils.parseToMillis(value);
            default:
                throw new RuntimeException("expected one of: " + EnumSet.of(JsonToken.VALUE_NULL, JsonToken.VALUE_STRING) + ", got: " + token + ", at " + parser.getCurrentLocation());
        }
    }

    private static String nextAttributeName(JsonParser parser) throws IOException {
        var token = parser.nextToken();
        if (token == JsonToken.VALUE_NULL) {
            return "";
        }

        currentToken(parser, JsonToken.START_OBJECT);
        String name = "";
        while (parser.nextToken() != JsonToken.END_OBJECT) {
            switch (parser.getCurrentName()) {
                case "name":
                    name = nextString(parser);
                    break;
                default:
                    skip(parser);
            }
        }
        return name;
    }

    private static void nextToken(JsonParser parser, JsonToken expected) throws IOException {
        JsonToken actual = parser.nextToken();
        if (actual != expected) {
            throw new RuntimeException("expected: " + expected + ", got: " + actual + ", at " + parser.getCurrentLocation());
        }
    }

    private static void currentToken(JsonParser parser, JsonToken expected) {
        JsonToken actual = parser.currentToken();
        if (actual != expected) {
            throw new RuntimeException("expected: " + expected + ", got: " + actual + ", at " + parser.getCurrentLocation());
        }
    }

    private static void skip(JsonParser parser) throws IOException {
        var token = parser.nextToken();
        switch (token) {
            case START_ARRAY:
            case START_OBJECT:
                parser.skipChildren();
        }
    }

    @Override
    public boolean hasNext() {
        try {
            var current = parser.getCurrentToken();
            if (current != null) {
                return true;
            }

            var next = parser.nextToken();
            return next != null;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Resource next() {
        try {
            if (!hasNext()) {
                throw new NoSuchElementException("End of stream");
            }

            currentToken(parser, JsonToken.START_OBJECT);
            Resource result = new Resource();
            result.name = "";
            boolean hasName = false;
            while (parser.nextToken() != JsonToken.END_OBJECT) {
                String name = parser.getCurrentName();
                switch (name) {
                    case "service":
                        result.service = nextString(parser);
                        break;
                    case "cloud_id":
                        result.cloudId = nextString(parser);
                        break;
                    case "folder_id":
                        result.folderId = nextString(parser);
                        break;
                    case "resource_id":
                        result.resourceId = nextString(parser);
                        break;
                    case "resource_type":
                        result.type = nextString(parser);
                        break;
                    case "timestamp":
                        result.updatedAt = nextTimestamp(parser);
                        break;
                    case "deleted":
                        result.deletedAt = nextTimestamp(parser);
                        break;
                    case "name":
                        result.name = nextString(parser);
                        hasName = true;
                        break;
                    case "reindex_timestamp":
                        result.reindexAt = nextTimestamp(parser);
                        break;
                    case "attributes":
                        if (!hasName) {
                            result.name = nextAttributeName(parser);
                        } else {
                            skip(parser);
                        }
                        break;
                    default:
                        skip(parser);
                }
            }

            parser.nextToken();
            return result;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws IOException {
        try {
            parser.close();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
