package ru.yandex.solomon.name.resolver.logbroker;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.google.common.base.Strings;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceValidator {
    private static final List<Validator> STRICT = List.of(
            ResourceValidator::validateCloudId,
            ResourceValidator::validateFolderId,
            ResourceValidator::validateService,
            ResourceValidator::validateType,
            ResourceValidator::validateResourceId,
            ResourceValidator::validateName,
            ResourceValidator::validateUpdatedAt,
            ResourceValidator::validateDeletedAt,
            ResourceValidator::validateReindexAt
    );

    private static final List<Validator> LIGHT = List.of(
            ResourceValidator::validateCloudId,
            ResourceValidator::validateService,
            ResourceValidator::validateResourceId,
            ResourceValidator::validateUpdatedAt,
            ResourceValidator::validateDeletedAt,
            ResourceValidator::validateReindexAt
    );

    public static ResourceValidator LIGHT_VALIDATOR = new ResourceValidator(LIGHT);
    public static ResourceValidator STRICT_VALIDATOR = new ResourceValidator(STRICT);
    private final List<Validator> validators;

    private ResourceValidator(List<Validator> validators) {
        this.validators = validators;
    }

    @Nullable
    public String validate(Resource resource) {
        for (var validator : validators) {
            var issue = validator.validate(resource);
            if (issue != null) {
                return issue;
            }
        }

        return null;
    }

    private static String validateCloudId(Resource resource) {
        return validateLength("cloudId", resource.cloudId);
    }

    private static String validateFolderId(Resource resource) {
        return validateLength("folderId", resource.folderId);
    }

    private static String validateService(Resource resource) {
        return validateLength("service", resource.service);
    }

    private static String validateType(Resource resource) {
        return validateLength("type", resource.type);
    }

    private static String validateResourceId(Resource resource) {
        return validateLength("resourceId", resource.resourceId);
    }

    private static String validateName(Resource resource) {
        if (Strings.isNullOrEmpty(resource.name)) {
            return null;
        }

        return validateLength("name", resource.name);
    }

    private static String validateUpdatedAt(Resource resource) {
        if (resource.updatedAt == 0) {
            return "updatedAt can not be zero";
        }

        return validateTs("updatedAt", resource.updatedAt);
    }

    private static String validateDeletedAt(Resource resource) {
        if (resource.deletedAt == 0) {
            return null;
        }

        if (resource.updatedAt < resource.deletedAt) {
            return "deletedAt " + prettyTs(resource.deletedAt) + " > updatedAt " + prettyTs(resource.updatedAt);
        }

        return validateTs("deletedAt", resource.deletedAt);
    }

    private static String validateReindexAt(Resource resource) {
        if (resource.reindexAt == 0) {
            return null;
        }

        return validateTs("reindexAt", resource.reindexAt);
    }

    @Nullable
    private static String validateLength(String fieldName, String str) {
        if (Strings.isNullOrEmpty(str)) {
            return fieldName + " absent";
        }

        if (str.length() >= 200) {
            return fieldName + " size " + str.length() + " >= 200";
        }

        return null;
    }

    @Nullable
    private static String validateTs(String fieldName, long ts) {
        if (ts == 0) {
            return null;
        }

        if (!InstantUtils.isGoodMillis(ts)) {
            return fieldName + " has invalid ts: " + prettyTs(ts);
        }

        long now = System.currentTimeMillis();
        long notAfter = now + TimeUnit.DAYS.toMillis(3);
        if (ts >= notAfter) {
            return fieldName + " has ts in future " + prettyTs(ts) + " now " + prettyTs(now);
        }

        return null;
    }

    private static String prettyTs(long ts) {
        if (ts == 0) {
            return "null";
        }

        return Instant.ofEpochMilli(ts).toString();
    }

    private interface Validator {
        @Nullable
        String validate(Resource resource);
    }
}
