package ru.yandex.solomon.name.resolver.sink;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardSink implements ResourceUpdater {
    private static final Logger logger = LoggerFactory.getLogger(ShardSink.class);
    private static final long BACKOFF_SLOT_MILLIS = 100;
    private static final int BACKOFF_CEILING = 11; // Slot 100ms, Max delay 51s

    final String shardId;
    private final ShardSinkClient client;
    private final ScheduledExecutorService timer;
    private final ShardSinkMetrics metrics;
    private final ActorWithFutureRunner actor;
    private final ArrayListLockQueue<ResourceUpdateRequest> queue = new ArrayListLockQueue<>();

    @Nullable
    private String node;
    private int attempt;
    private List<ResourceUpdateRequest> inFlight = List.of();

    // for manager ui
    Throwable lastError;
    Instant lastErrorTime = Instant.EPOCH;
    ScheduledFuture scheduled;

    public ShardSink(String shardId, ShardSinkClient client, Executor executor, ScheduledExecutorService timer, ShardSinkMetrics metrics) {
        this.shardId = shardId;
        this.client = client;
        this.timer = timer;
        this.metrics = metrics;
        this.actor = new ActorWithFutureRunner(this::act, executor);
    }

    @Override
    public void update(ResourceUpdateRequest request) {
        metrics.queueSize.add(1);
        queue.enqueue(request);
        actor.schedule();
    }

    private CompletableFuture<Void> act() {
        var doneFuture = new CompletableFuture<Void>();
        tryUpdate().whenComplete((ignore, e) -> {
            if (e != null) {
                metrics.errors.inc();
                lastError = e;
                lastErrorTime = Instant.now();
                long delayMillis = backoffTimeMillis(attempt++);
                logger.error("shard sink {} failed, retry after {} ms", shardId, delayMillis, e);
                scheduled = timer.schedule(() -> {
                    // release actor, try act one more time
                    actor.schedule();
                    doneFuture.complete(null);
                }, delayMillis, TimeUnit.MILLISECONDS);
            } else {
                attempt = 0;
                doneFuture.complete(null);
            }
        });
        return doneFuture;
    }

    private CompletableFuture<?> tryUpdate() {
        try {
            if (node == null) {
                return resolveNode();
            }

            if (inFlight.isEmpty()) {
                inFlight = queue.dequeueAll();
                metrics.queueSize.add(-inFlight.size());
                metrics.resourceInFlight.set(inFlight.size());
            }

            if (inFlight.isEmpty()) {
                return completedFuture(null);
            }

            return updateOnNode();
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private CompletableFuture<?> resolveNode() {
        return client.resolveNode(shardId)
                .whenComplete((node, e) -> {
                    if (e != null) {
                        logger.error("unable resolve node for shard {}", shardId, e);
                        return;
                    }

                    this.node = node;
                    this.actor.schedule();
                });
    }

    private CompletableFuture<Void> updateOnNode() {
        List<Resource> resources = new ArrayList<>(inFlight.size());
        for (var req : inFlight) {
            resources.add(req.resource());
        }

        return client.update(node, shardId, resources, false, "")
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        var status = Status.fromThrowable(e);
                        logger.error("update resources for shard {} at node {} failed {}", shardId, node, status.getCode());
                        invalidateNodeIfNecessary(status);
                    } else {
                        actor.schedule();
                        var requests = inFlight;
                        inFlight = List.of();
                        metrics.resourceInFlight.set(0);
                        for (var req : requests) {
                            req.complete();
                        }
                    }
                });
    }

    private void invalidateNodeIfNecessary(Status error) {
        switch (error.getCode()) {
            case NOT_FOUND:
            case DEADLINE_EXCEEDED:
            case UNAVAILABLE:
                node = null;
        }
    }

    private long backoffTimeMillis(int attempt) {
        int slots = 1 << Math.min(attempt, BACKOFF_CEILING);
        return randomizeDelay(BACKOFF_SLOT_MILLIS * slots);
    }

    private long randomizeDelay(long delayMillis) {
        if (delayMillis == 0) {
            return 0;
        }

        var half = delayMillis / 2;
        return half + ThreadLocalRandom.current().nextLong(half);
    }
}
