package ru.yandex.solomon.name.resolver.sink;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import io.grpc.MethodDescriptor;
import io.grpc.Status;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.balancer.BalancerServiceGrpc;
import ru.yandex.solomon.balancer.TCreateShardRequest;
import ru.yandex.solomon.balancer.TCreateShardResponse;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.grpc.Proto;
import ru.yandex.solomon.name.resolver.protobuf.ResourceServiceGrpc;
import ru.yandex.solomon.name.resolver.protobuf.UpdateResourcesRequest;

import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardSinkClientImpl implements ShardSinkClient {
    private static final long MAX_MESSAGE_SIZE = 20 << 20; // 20 Mib
    private final DistributedLock leader;
    private final ClusterDiscovery<GrpcTransport> discovery;

    public ShardSinkClientImpl(DistributedLock leader, ClusterDiscovery<GrpcTransport> discovery) {
        this.leader = leader;
        this.discovery = discovery;
    }

    @Override
    public CompletableFuture<String> resolveNode(String shardId) {
        var opt = leader.lockDetail();
        if (opt.isEmpty()) {
            return failedFuture(Status.NOT_FOUND
                    .withDescription("Unknown leader")
                    .asRuntimeException());
        }

        var req = TCreateShardRequest.newBuilder()
                .setShardId(shardId)
                .build();

        var leader = opt.get().owner();
        return unaryCall(leader, BalancerServiceGrpc.getCreateShardMethod(), req)
                .thenApply(TCreateShardResponse::getNode);
    }

    @Override
    public CompletableFuture<Void> update(String node, String shardId, List<Resource> resources, boolean removeOtherOfService, String serviceProviderId) {
        List<CompletableFuture<?>> futures = new ArrayList<>();

        int bytes = 0;
        var req = prepareUpdateReq(shardId);
        for (var resource : resources) {
            var proto = Proto.toProto(resource);
            req.addResources(proto);
            bytes += proto.getSerializedSize();
            if (bytes >= MAX_MESSAGE_SIZE) {
                futures.add(update(node, req.build()));
                bytes = 0;
                req = prepareUpdateReq(shardId);
            }
            req.setRemoveOtherOfService(removeOtherOfService);
            req.setServiceProviderId(serviceProviderId);
        }

        if (req.getResourcesCount() > 0) {
            futures.add(update(node, req.build()));
        }

        return CompletableFutures.allOfVoid(futures);
    }

    private UpdateResourcesRequest.Builder prepareUpdateReq(String shardId) {
        return UpdateResourcesRequest.newBuilder().setCloudId(shardId);
    }

    private CompletableFuture<?> update(String node, UpdateResourcesRequest req) {
        return unaryCall(node, ResourceServiceGrpc.getUpdateResourcesMethod(), req);
    }

    private <ReqT, RespT> CompletableFuture<RespT> unaryCall(String target, MethodDescriptor<ReqT, RespT> method, ReqT request) {
        try {
            GrpcTransport transport = discovery.getTransportByNode(target);
            return transport.unaryCall(method, request, 0);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }
}
