package ru.yandex.solomon.name.resolver.sink;

import java.time.Instant;
import java.util.Comparator;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.find.annotation.NamedObjectFinderAnnotation;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.solomon.util.ExceptionUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardSinkDispatcher implements ResourceUpdater {
    private final ShardSinkClient client;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    private final ShardSinkMetrics metrics;
    private final Cache<String, ShardSink> sinkByShard;

    public ShardSinkDispatcher(ShardSinkClient client, ExecutorService executor, ScheduledExecutorService timer, ShardSinkMetrics metrics) {
        this.client = client;
        this.executor = executor;
        this.timer = timer;
        this.metrics = metrics;
        this.sinkByShard = CacheBuilder.newBuilder()
                .expireAfterAccess(1, TimeUnit.HOURS)
                .removalListener(notification -> this.metrics.shardCount.add(-1))
                .build();
    }

    @NamedObjectFinderAnnotation
    public ShardSink getShardSinkById(String shardId) {
        return sinkByShard.getIfPresent(shardId);
    }

    @Override
    public void update(ResourceUpdateRequest request) {
        try {
            String shardId = request.resource().cloudId;
            var sink = sinkByShard.get(shardId, () -> {
                metrics.shardCount.add(1);
                return new ShardSink(shardId, client, executor, timer, metrics);
            });
            sink.update(request);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    @ExtraContent("Shards")
    public void shardsPage(ExtraContentParam p) {
        var records = sinkByShard.asMap()
                .values()
                .stream()
                .map(Record::new)
                .sorted(Comparator.<Record>naturalOrder().reversed())
                .collect(Collectors.toList());

        var hw = p.getHtmlWriter();
        hw.tableTable(() -> {
            hw.trThs("Id", "Latest Error Time", "Latest Error");
            for (var record : records) {
                hw.tr(() -> {
                    hw.td(() -> hw.aHref(namedObjectLink(new NamedObjectId(ShardSink.class, record.shardId)), record.shardId));
                    hw.td(() -> {
                        if (record.errorTime.equals(Instant.EPOCH)) {
                            hw.write("none");
                        } else {
                            hw.write(record.errorTime.toString());
                        }
                    });
                    hw.td(() -> {
                        if (record.error != null) {
                            hw.preText(ExceptionUtils.printStackTrace(record.error));
                        } else {
                            hw.write("none");
                        }
                    });
                });
            }
        });
    }

    private static class Record implements Comparable<Record> {
        private String shardId;
        private Throwable error;
        private Instant errorTime;

        public Record(ShardSink shard) {
            this.shardId = shard.shardId;
            this.error = shard.lastError;
            this.errorTime = shard.lastErrorTime;
        }

        @Override
        public int compareTo(Record that) {
            return errorTime.compareTo(that.errorTime);
        }
    }
}
