package ru.yandex.solomon.name.resolver.spring;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.nio.file.Path;
import java.time.Clock;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.balancer.BalancerHolderImpl;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.balancer.TotalShardCounter;
import ru.yandex.solomon.balancer.TotalShardCounterImpl;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.remote.RemoteCluster;
import ru.yandex.solomon.balancer.remote.RemoteClusterImpl;
import ru.yandex.solomon.balancer.remote.RemoteNodeClient;
import ru.yandex.solomon.balancer.remote.RemoteNodeClientImpl;
import ru.yandex.solomon.balancer.www.BalancerController;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.config.protobuf.name.resolver.TNameResolverBalancerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.locks.LockServiceImpl;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.name.resolver.NameResolverShardFactory;
import ru.yandex.solomon.name.resolver.balancer.AssignmentGate;
import ru.yandex.solomon.name.resolver.balancer.AssignmentStore;
import ru.yandex.solomon.name.resolver.balancer.AssignmentSummary;
import ru.yandex.solomon.name.resolver.balancer.GrpcBalancerService;
import ru.yandex.solomon.name.resolver.balancer.NameResolverResource;
import ru.yandex.solomon.name.resolver.balancer.NameResolverShardsHolder;
import ru.yandex.solomon.name.resolver.db.ShardsDao;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.util.file.SimpleFileStorage;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
public class BalancerContext {
    private static final Path CACHE_DIR = Path.of("cache");

    private final Clock clock;
    private final ScheduledExecutorService timer;
    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;

    public BalancerContext(Clock clock, ThreadPoolProvider threads, MetricRegistry registry) {
        this.clock = clock;
        this.timer = threads.getSchedulerExecutorService();
        this.threads = threads;
        this.registry = registry;
    }

    @Bean
    public LockService lockService(LocksDao locksDao) {
        return new LockServiceImpl(HostUtils.getFqdn(), locksDao, clock, timer, registry);
    }

    @Bean
    @BalancerLock
    public DistributedLock balancerLock(LockService lockService) {
        return lockService.distributedLock("Balancer");
    }

    @Bean
    public TotalShardCounter totalShardCounter() {
        return new TotalShardCounterImpl();
    }

    @Bean
    public RemoteNodeClient remoteNodeClient(
        ClusterDiscovery<GrpcTransport> discovery,
        TotalShardCounter shardCounter)
    {
        return new RemoteNodeClientImpl(discovery, shardCounter);
    }

    @Bean
    public RemoteCluster remoteCluster(RemoteNodeClient remoteNodeClient) {
        return new RemoteClusterImpl(remoteNodeClient, timer, clock);
    }

    @Bean
    public NameResolverShardsHolder shardsHolder(ShardsDao shardsDao, @BalancerLock DistributedLock lock) {
        return new NameResolverShardsHolder(shardsDao, lock);
    }

    @Bean
    public BalancerHolderImpl balancerHolder(
            TNameResolverBalancerConfig config,
            @BalancerLock DistributedLock lock,
            RemoteCluster cluster,
            ShardsHolder shards,
            TotalShardCounter shardCounter,
            BalancerDao dao)
    {
        var executor = threads.getExecutorService(config.getThreadPoolName(), "BalancerConfig.ThreadPoolName");
        return new BalancerHolderImpl(
            clock,
            lock,
            NameResolverResource.RESOURCES,
            cluster,
            executor,
            timer,
            shards,
            shardCounter,
            dao);
    }

    @Bean
    public Bootstrap balanerBootstrap(BalancerHolderImpl balancer) {
        return new Bootstrap(balancer);
    }

    @Bean
    public BalancerController balancerController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            BalancerHolder balancer,
            HttpServerConfig config)
    {
        return new BalancerController(authenticator, authorizer, balancer, NameResolverShard.class, config.getPort());
    }

    @Bean
    public RootLink balancerLink() {
        return new RootLink("/balancer", "Balancer");
    }

    @Bean
    public AssignmentGate assignmentGate(NameResolverLocalShards shards, @BalancerLock DistributedLock lock, NameResolverShardFactory factory) {
        return new AssignmentGate(shards, lock, factory);
    }

    @Bean
    public AssignmentStore assignmentStore(NameResolverLocalShards shards, NameResolverShardFactory factory, TNameResolverBalancerConfig config) {
        var executor = threads.getExecutorService(config.getThreadPoolName(), "BalancerConfig.ThreadPoolName");
        var storage = new SimpleFileStorage(CACHE_DIR);
        return new AssignmentStore(storage, shards, executor, factory);
    }

    @Bean
    public AssignmentSummary assignmentSummary(Clock clock, NameResolverLocalShards shards) {
        return new AssignmentSummary(clock, shards);
    }

    @Bean
    public GrpcBalancerService balancerService(AssignmentGate gate, AssignmentStore store, AssignmentSummary summary, BalancerHolder balancer) {
        return new GrpcBalancerService(gate, store, summary, balancer);
    }

    @Bean
    public GrpcBalancerService.Proxy balancerServiceProxy(GrpcBalancerService service) {
        return service.createProxy();
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface BalancerLock {
    }

    @ParametersAreNonnullByDefault
    private static class Bootstrap implements ApplicationListener<ContextRefreshedEvent> {
        private final BalancerHolderImpl balancer;

        public Bootstrap(BalancerHolderImpl balancer) {
            this.balancer = balancer;
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent event) {
            balancer.start();
        }
    }
}
