package ru.yandex.solomon.name.resolver.spring;

import java.util.Optional;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.GrpcClusterDiscovery;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.name.resolver.TNameResolverDiscoveryConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
        GrpcConfigurationContext.class
})
public class DiscoveryContext {
    private final String clientId;
    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;

    public DiscoveryContext(@Qualifier("ClientId") Optional<String> clientId, ThreadPoolProvider threads, MetricRegistry registry) {
        this.clientId = clientId.map(Strings::emptyToNull).orElse("name-resolver-server");
        this.threads = threads;
        this.registry = registry;
    }

    @Bean
    public ClusterDiscovery<GrpcTransport> clusterDiscovery(
            TNameResolverDiscoveryConfig config,
            ClientOptionsFactory clientOptionsFactory)
    {
        var addresses = config.getGrpcConfig().getAddressesList();
        var opts = clientOptionsFactory.newBuilder("", config.getGrpcConfig())
            .setClientId(clientId)
            .setMetricRegistry(registry)
            .build();
        var executor = threads.getExecutorService(config.getThreadPoolName(), "DiscoveryConfig.ThreadPoolName");
        var timer = threads.getSchedulerExecutorService();
        var refreshMillis = TimeUnitConverter.millis(config.getRefreshInterval());
        return GrpcClusterDiscovery.of(opts, addresses, timer, executor, refreshMillis);
    }
}
