package ru.yandex.solomon.name.resolver.spring;

import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.ContextRefreshedEvent;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TIamTokenClientConfig;
import ru.yandex.solomon.config.protobuf.TLogbrokerInstanceConfig;
import ru.yandex.solomon.config.protobuf.name.resolver.TNameResolverConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.name.resolver.IssueTracker;
import ru.yandex.solomon.name.resolver.logbroker.PersqueueClientFactory;
import ru.yandex.solomon.name.resolver.logbroker.PersqueueInstance;
import ru.yandex.solomon.name.resolver.logbroker.PersqueueReader;
import ru.yandex.solomon.name.resolver.logbroker.ResourceFilter;
import ru.yandex.solomon.name.resolver.sink.ResourceUpdater;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.ydb.YdbAuthProviders;

/**
 * @author Vladimir Gordiychuk
 */
@Import(YdbAuthContext.class)
@Configuration
public class LogbrokerContext {
    private static final Logger logger = LoggerFactory.getLogger(LogbrokerContext.class);

    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;
    private final List<TLogbrokerInstanceConfig> config;

    public LogbrokerContext(ThreadPoolProvider threads, MetricRegistry registry, TNameResolverConfig config) {
        this.threads = threads;
        this.registry = registry;
        this.config = config.getLogbrokerInstanceList();
    }

    @Bean
    @ConditionalOnBean(TIamTokenClientConfig.class)
    public List<PersqueueInstance> persqueueInstances(YdbAuthProviders auth, ResourceUpdater updater, IssueTracker issueTracker, ResourceFilter filter) {
        return config.stream()
                .map(instanceConfig -> {
                    var clientConfig = instanceConfig.getClientConfig();
                    var client = PersqueueClientFactory.makeClient(auth, threads, registry, clientConfig);
                    var name = clientConfig.getName();
                    var reader = new PersqueueReader(name, client, issueTracker, threads, updater, filter, instanceConfig.getReaderConfig(), registry);
                    return new PersqueueInstance(client, reader);
                })
                .collect(Collectors.toList());
    }

    @Bean
    @ConditionalOnBean(TIamTokenClientConfig.class)
    public Bootstrap logbrokerBootstrap(List<PersqueueInstance> instances, ResourceFilter filter) {
        return new Bootstrap(instances, filter, threads.getSchedulerExecutorService());
    }

    @ParametersAreNonnullByDefault
    private static class Bootstrap implements AutoCloseable, ApplicationListener<ContextRefreshedEvent> {
        private final List<PersqueueInstance> instances;
        private final ResourceFilter filter;
        private final ScheduledExecutorService timer;

        public Bootstrap(List<PersqueueInstance> instances, ResourceFilter filter, ScheduledExecutorService timer) {
            this.instances = instances;
            this.filter = filter;
            this.timer = timer;
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent event) {
            if (filter.isReady()) {
                instances.forEach(PersqueueInstance::start);
            } else {
                logger.warn("Resource filter not ready yet, skip start topic consuming");
                timer.schedule(() -> onApplicationEvent(event), 1, TimeUnit.SECONDS);
            }
        }

        @Override
        public void close() {
            instances.forEach(PersqueueInstance::close);
        }
    }
}
