package ru.yandex.solomon.name.resolver.spring;

import java.time.Clock;
import java.util.List;
import java.util.function.Predicate;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.config.protobuf.name.resolver.TNameResolverFilterConfig;
import ru.yandex.solomon.config.protobuf.name.resolver.TNameResolverLocalShardConfig;
import ru.yandex.solomon.config.protobuf.name.resolver.TNameResolverTtlConfig;
import ru.yandex.solomon.config.thread.LazyThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.ctx.ServiceAuthContext;
import ru.yandex.solomon.http.filters.HttpFirewallFilter;
import ru.yandex.solomon.http.filters.RequestLoggingFilter;
import ru.yandex.solomon.name.resolver.GlobalShardMetrics;
import ru.yandex.solomon.name.resolver.GrpcResourceService;
import ru.yandex.solomon.name.resolver.IssueTracker;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShardFactory;
import ru.yandex.solomon.name.resolver.NameResolverShardFactoryImpl;
import ru.yandex.solomon.name.resolver.ResourceService;
import ru.yandex.solomon.name.resolver.ResourceServiceImpl;
import ru.yandex.solomon.name.resolver.ServiceProviderHolderImpl;
import ru.yandex.solomon.name.resolver.ServiceProviderListener;
import ru.yandex.solomon.name.resolver.ServiceProviderWatcher;
import ru.yandex.solomon.name.resolver.ShardMetricSupplier;
import ru.yandex.solomon.name.resolver.db.ResourcesDao;
import ru.yandex.solomon.name.resolver.handler.ServerStatusHandler;
import ru.yandex.solomon.name.resolver.handler.UpdateResourcesHandler;
import ru.yandex.solomon.name.resolver.logbroker.ResourceFilter;
import ru.yandex.solomon.name.resolver.sink.ShardSinkClient;
import ru.yandex.solomon.name.resolver.ttl.TtlScheduler;
import ru.yandex.solomon.name.resolver.ttl.TtlSchedulerOpts;
import ru.yandex.solomon.name.resolver.www.NameResolverLocalShardsWww;
import ru.yandex.solomon.name.resolver.www.ResourcesController;
import ru.yandex.solomon.selfmon.GeneralMonitoringContext;
import ru.yandex.solomon.selfmon.mon.SelfMetricsController;
import ru.yandex.solomon.staffOnly.StaffOnlyController;
import ru.yandex.solomon.staffOnly.StaffOnlyRedirectController;

import static ru.yandex.solomon.config.OptionalSet.setInt;
import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
@Import({
        RequestLoggingFilter.class,
        HttpFirewallFilter.class,
        SelfMetricsController.class,
        GeneralMonitoringContext.class,
        LazyThreadPoolProvider.class,
        ServiceAuthContext.class,
        StaffOnlyController.class,
        StaffOnlyRedirectController.class,
        LogbrokerContext.class,
        YdbContext.class,
        NameResolverLocalShardsWww.class,
        ResourcesController.class,
        DiscoveryContext.class,
        BalancerContext.class,
        GrpcServerContext.class,
        SinkContext.class,
        GeneralMonitoringContext.class,
        IssueTrackerContext.class,
})
@Configuration
public class NameResolverContext {

    @Bean
    public Clock clock() {
        return Clock.systemUTC();
    }

    @Bean
    public NameResolverLocalShards localShards() {
        return new NameResolverLocalShards();
    }

    @Bean
    public NameResolverShardFactory shardFactory(ResourcesDao resourcesDao, IssueTracker issueTracker, ThreadPoolProvider threads, GlobalShardMetrics metrics, TNameResolverLocalShardConfig config) {
        var executor = threads.getExecutorService(config.getThreadPoolName(), "LocalShardConfig.ThreadPoolName");
        var timer = threads.getSchedulerExecutorService();
        return new NameResolverShardFactoryImpl(resourcesDao, issueTracker, executor, metrics, timer);
    }

    @Bean
    public ResourceService resourceService(NameResolverLocalShards shards) {
        return new ResourceServiceImpl(shards);
    }

    @Bean
    public ServerStatusHandler statusHandler(BalancerHolder balancerHolder, ClusterDiscovery<GrpcTransport> discovery) {
        return new ServerStatusHandler(balancerHolder, discovery);
    }

    @Bean
    public UpdateResourcesHandler updateHandler(ResourceService resourceService) {
        return new UpdateResourcesHandler(resourceService);
    }

    @Bean
    public GrpcResourceService grpcResourceService(ServerStatusHandler status, ResourceService resourceService, ShardSinkClient shardSinkClient) {
        return new GrpcResourceService(status, resourceService, shardSinkClient);
    }

    @Bean
    public GlobalShardMetrics globalResourceMetrics() {
        return new GlobalShardMetrics();
    }

    @Bean
    public ShardMetricSupplier resourceMetricSupplier(GlobalShardMetrics global, NameResolverLocalShards shards) {
        return new ShardMetricSupplier(global, shards);
    }

    @Bean
    public ServiceProviderHolderImpl serviceProviderHolder() {
        return new ServiceProviderHolderImpl();
    }

    @Bean
    public ServiceProviderWatcher serviceProviderWatcher(ThreadPoolProvider threads, ServiceProvidersDao dao, List<ServiceProviderListener> listeners) {
        var executor = threads.getExecutorService("CpuLowPriority", "");
        var timer = threads.getSchedulerExecutorService();
        return new ServiceProviderWatcher(dao, listeners, executor, timer);
    }

    @Bean
    public TtlScheduler ttlScheduler(
            TNameResolverTtlConfig ttlConfig,
            NameResolverLocalShards shards,
            IssueTracker tracker,
            ThreadPoolProvider threads,
            ResourceFilter filter,
            MetricRegistry registry,
            Clock clock)
    {
        var opts = TtlSchedulerOpts.newBuilder();
        setInt(opts::maxInFlight, ttlConfig.getMaxTtlTaskInFlight());
        setTime(opts::resourceTtl, ttlConfig.getResourceTtl());
        setTime(opts::firstRunPeriod, ttlConfig.getFirsRunPeriod());
        setTime(opts::failRunPeriod, ttlConfig.getFailRunPeriod());
        setTime(opts::successRunPeriod, ttlConfig.getSuccessRunPeriod());
        opts.lightValidations(Boolean.TRUE.equals(ttlConfig.getLightValidators()));

        var executor = threads.getExecutorService(ttlConfig.getThreadPoolName(), "TtlConfig.ThreadPoolName");
        var timer = threads.getSchedulerExecutorService();
        var deletePredicate = Predicate.not(filter);
        return new TtlScheduler(opts.build(), shards, tracker, deletePredicate, registry, executor, clock, timer);
    }

    @Bean
    public ResourceFilter resourceFilter(TNameResolverFilterConfig config) {
        return new ResourceFilter(config.getEnable());
    }
}
