package ru.yandex.solomon.name.resolver.spring;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.time.Clock;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.dao.YdbBalancerDaoV2;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceProvidersDao;
import ru.yandex.solomon.ctx.ServiceAuthContext;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.name.resolver.db.ResourcesDao;
import ru.yandex.solomon.name.resolver.db.ShardsDao;
import ru.yandex.solomon.name.resolver.db.ydb.YdbResourcesDao;
import ru.yandex.solomon.name.resolver.db.ydb.YdbShardsDao;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.ydb.YdbAuthProviders;
import ru.yandex.solomon.ydb.YdbClients;

/**
 * @author Vladimir Gordiychuk
 */
@Import({
        YdbAuthContext.class,
        ServiceAuthContext.class
})
@Configuration
public class YdbContext {
    private static final String ROOT_DIR = "/Name-Resolver/V1";
    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;

    public YdbContext(ThreadPoolProvider threads, MetricRegistry registry) {
        this.threads = threads;
        this.registry = registry;
    }

    @Bean
    @LocalYdb
    public YdbClients localYdbClients(
            @Qualifier("KikimrClientConfig") TKikimrClientConfig config,
            Optional<YdbAuthProviders> authProvidersOpt,
            Optional<IamTokenClient> iamTokenClient,
            SecretProvider secrets)
    {
        if (authProvidersOpt.isPresent()) {
            return new YdbClients("localYdb", config, threads, registry, authProvidersOpt.get());
        }
        return new YdbClients("KikimrClientConfig", config, threads, registry, iamTokenClient, secrets);
    }

    @Bean
    public MetricSupplier localYdbTableClientMetrics(@LocalYdb YdbClients ydb) {
        return ydb.getTableClientMetrics();
    }

    @Bean
    @LockYdb
    public YdbClients locksYdbClients(
            @Qualifier("KikimrClientConfig") TKikimrClientConfig config,
            Optional<YdbAuthProviders> authProvidersOpt,
            Optional<IamTokenClient> iamTokenClient,
            SecretProvider secrets)
    {
        if (authProvidersOpt.isPresent()) {
            return new YdbClients("lockYdb", config, threads, registry, authProvidersOpt.get());
        }
        return new YdbClients("KikimrClientConfig", config, threads, registry, iamTokenClient, secrets);
    }

    @Bean
    public MetricSupplier lockYdbTableClientMetrics(@LocalYdb YdbClients ydb) {
        return ydb.getTableClientMetrics();
    }

    @Bean
    @GlobalYdb
    public YdbClients globalYdbClients(
            @Qualifier("CrossDcKikimrClientConfig") TKikimrClientConfig config,
            Optional<YdbAuthProviders> authProvidersOpt,
            Optional<IamTokenClient> iamTokenClient,
            SecretProvider secrets)
    {
        if (authProvidersOpt.isPresent()) {
            return new YdbClients("globalYdb", config, threads, registry, authProvidersOpt.get());
        }
        return new YdbClients("CrossDcKikimrClientConfig", config, threads, registry, iamTokenClient, secrets);
    }

    @Bean
    public MetricSupplier globalYdbTableClientMetrics(@GlobalYdb YdbClients ydb) {
        return ydb.getTableClientMetrics();
    }

    @Bean
    public ResourcesDao ydbResourcesDao(@LocalYdb YdbClients ydb) {
        var dao = new YdbResourcesDao(ydb.getSchemaRoot() + ROOT_DIR, ydb.getTableClient(), ydb.getSchemeClient(), new ObjectMapper());
        dao.createSchemaForTests();
        dao.migrateSchema();
        return measure(dao, ResourcesDao.class);
    }

    @Bean
    public LocksDao locksDao(Clock clock, @LockYdb YdbClients ydb) {
        var path = ydb.getSchemaRoot() + ROOT_DIR;
        var dao = new YdbLocksDao(ydb.getTableClient(), clock, path);
        dao.createSchema();
        return measure(dao, LocksDao.class);
    }

    @Bean
    public ShardsDao shardsDao(@LocalYdb YdbClients ydb) {
        var dao = new YdbShardsDao(ydb.getSchemaRoot() + ROOT_DIR, ydb.getTableClient(), ydb.getSchemeClient());
        dao.createSchemaForTests();
        return measure(dao, ShardsDao.class);
    }

    @Bean
    public BalancerDao balancerDao(@LocalYdb YdbClients ydb) {
        var rootPath = ydb.getSchemaRoot() + ROOT_DIR;
        var v2 = new YdbBalancerDaoV2(rootPath, ydb.getTableClient(), ydb.getSchemeClient());
        v2.createSchema();
        return measure(v2, BalancerDao.class);
    }

    @Bean
    public ServiceProvidersDao ydbServiceProviders(@GlobalYdb YdbClients ydb) {
        var path = ydb.getSchemaRoot() + "/Config/V2/ServiceProvider";
        var dao = new YdbServiceProvidersDao(ydb.getTableClient(), path, new ObjectMapper());
        dao.migrateSchema();
        return measure(dao, ServiceProvidersDao.class);
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return DaoMetricsProxy.of(dao, daoClass, registry);
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface LocalYdb {
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface LockYdb {
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface GlobalYdb {
    }
}
