package ru.yandex.solomon.name.resolver.stats;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.solomon.name.resolver.client.Resource;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceUpdateStats {
    public final Map<ResourceKey, ResourceStats> statsByKey;

    public ResourceUpdateStats(int capacity) {
        statsByKey = new HashMap<>(capacity);
    }

    public static ResourceUpdateStats of(List<Resource> resources) {
        ResourceUpdateStats result = new ResourceUpdateStats(resources.size());
        for (var resource : resources) {
            var stats = result.getOrCreateStats(resource);
            if (resource.deletedAt != 0) {
                stats.countDeleted++;
            } else {
                stats.countAlive++;
            }
        }
        return result;
    }

    public void receiveAdd(Resource resource) {
        var stats = getOrCreateStats(resource);
        if (resource.deletedAt != 0) {
            stats.countDeleted++;
            stats.receiveDelete++;
        } else {
            stats.countAlive++;
            stats.receiveAdd++;
        }
    }

    public void receiveObsolete(Resource resource) {
        var stats = getOrCreateStats(resource);
        stats.receiveObsolete++;
    }

    public void receiveReindex(Resource resource) {
        var stats = getOrCreateStats(resource);
        stats.receiveReindex++;
    }

    public void receiveUpdate(Resource prev, Resource update) {
        decrementCount(prev);
        var stats = getOrCreateStats(update);
        if (update.deletedAt != 0) {
            stats.receiveDelete++;
            stats.countDeleted++;
        } else {
            stats.receiveUpdate++;
            stats.countAlive++;
        }
    }

    public void receiveLost(Resource resource) {
        var stats = getOrCreateStats(resource);
        if (resource.deletedAt != 0) {
            stats.countDeleted++;
        } else {
            stats.countAlive++;
        }
        stats.receiveLost++;
    }

    public void receiveLost(Resource prev, Resource update) {
        decrementCount(prev);
        var stats = getOrCreateStats(update);
        if (update.deletedAt != 0) {
            stats.countDeleted++;
        } else {
            stats.countAlive++;
        }

        stats.receiveLost++;
    }

    public void receiveRemoveFromDb(Resource resource) {
        var stats = getOrCreateStats(resource);
        if (resource.deletedAt != 0) {
            stats.countDeleted--;
        } else {
            stats.countAlive--;
        }
        stats.receiveRemoveFromDb++;
    }

    private void decrementCount(Resource resource) {
        var stats = getOrCreateStats(resource);
        if (resource.deletedAt != 0) {
            stats.countDeleted--;
        } else {
            stats.countAlive--;
        }
    }

    private ResourceStats getOrCreateStats(Resource resource) {
        var key = ResourceKey.of(resource);
        var result = statsByKey.get(key);
        if (result == null) {
            result = new ResourceStats();
            statsByKey.put(key, result);
        }
        return result;
    }
}
