package ru.yandex.solomon.name.resolver.ttl;

import java.util.function.Predicate;

import it.unimi.dsi.fastutil.objects.Object2LongMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.logbroker.ResourceValidator;

import static com.google.common.base.Preconditions.checkArgument;


/**
 * @author Vladimir Gordiychuk
 */
public class TtlPredicate implements Predicate<Resource> {
    private final Logger logger = LoggerFactory.getLogger(TtlPredicate.class);

    private final long now;
    private final long ttlMillis;
    private final Object2LongMap<String> metricsTtlByServiceProvider;
    private final ResourceValidator resourceValidator;

    public TtlPredicate(long now, long ttlMillis, Object2LongMap<String> metricsTtlByServiceProvider, ResourceValidator resourceValidator) {
        this.resourceValidator = resourceValidator;
        checkArgument(ttlMillis > 0);
        this.now = now;
        this.ttlMillis = ttlMillis;
        this.metricsTtlByServiceProvider = metricsTtlByServiceProvider;
    }

    /**
     * @return true if resource should be deleted, otherwise return false
     */
    @Override
    public boolean test(Resource resource) {
        if (isNotValid(resource)) {
            return true;
        }

        if (resource.deletedAt == 0) {
            return false;
        }

        if (!metricsTtlByServiceProvider.containsKey(resource.service)) {
            // unknown service provider
            return now - resource.deletedAt >= ttlMillis;
        }

        long metricsTtl = metricsTtlByServiceProvider.get(resource.service);
        if (metricsTtl == 0) {
            return false;
        }

        if (resource.replaced) {
            return now - resource.deletedAt >= ttlMillis;
        }

        return now - resource.deletedAt >= ttlMillis + metricsTtl;
    }

    private boolean isNotValid(Resource resource) {
        var issue = resourceValidator.validate(resource);
        if (issue != null) {
            logger.warn("invalid: {}, issue {}", resource, issue);
            return true;
        }

        return false;
    }
}
