package ru.yandex.solomon.name.resolver.ttl;

import java.time.Clock;
import java.util.Comparator;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.objects.Object2LongMap;
import it.unimi.dsi.fastutil.objects.Object2LongOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.name.resolver.IssueTracker;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.name.resolver.ServiceProviderListener;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.logbroker.ResourceValidator;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Vladimir Gordiychuk
 */
public class TtlScheduler implements ServiceProviderListener, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(TtlScheduler.class);

    private final TtlSchedulerOpts opts;
    private final NameResolverLocalShards shards;
    private final IssueTracker issueTracker;
    private final Predicate<Resource> deletePredicate;
    private final Metrics metrics;
    private final Executor executor;
    private final Clock clock;
    private final ActorRunner actor;
    private final ScheduledFuture<?> pereodicallyAct;
    private final ConcurrentMap<String, Status> statusByShardId = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, TtlTask> activeTask = new ConcurrentHashMap<>();
    @Nullable
    private volatile Object2LongOpenHashMap<String> metricsTtlByService;
    private volatile CountDownLatch actSync = new CountDownLatch(1);
    private volatile boolean closed;

    public TtlScheduler(
            TtlSchedulerOpts opts,
            NameResolverLocalShards shards,
            IssueTracker issueTracker,
            Predicate<Resource> deletePredicate,
            MetricRegistry registry,
            Executor executor,
            Clock clock,
            ScheduledExecutorService timer)
    {
        this.opts = opts;
        this.shards = shards;
        this.issueTracker = issueTracker;
        this.deletePredicate = deletePredicate;
        this.metrics = new Metrics(registry);
        this.executor = executor;
        this.clock = clock;
        this.actor = new ActorRunner(this::act, executor);
        long jitter = ThreadLocalRandom.current().nextLong(opts.schedulePeriodMs);
        this.pereodicallyAct = timer.scheduleAtFixedRate(actor::schedule, jitter, opts.schedulePeriodMs, TimeUnit.MILLISECONDS);
    }

    private void act() {
        if (closed) {
            return;
        }

        var metricsTtlByService = this.metricsTtlByService;
        if (metricsTtlByService == null) {
            return;
        }

        long allowSchedule = opts.maxInFlight - metrics.taskMetrics.getInFlight();
        if (allowSchedule <= 0) {
            return;
        }

        long now = clock.millis();
        shards.stream()
                .filter(shard -> isReadyForTtl(now, shard))
                .sorted(Comparator.comparingLong(o -> statusByShardId.get(o.cloudId).createdAtMillis))
                .limit(allowSchedule)
                .forEach(shard -> {
                    runTtlTask(shard, metricsTtlByService, now);
                });
        actSync.countDown();
    }

    CountDownLatch actSync() {
        var sync = new CountDownLatch(1);
        actSync = sync;
        actor.schedule();
        return sync;
    }

    @Nullable
    public Status getTtlStatus(String shardId) {
        return statusByShardId.get(shardId);
    }

    private boolean isReadyForTtl(long now, NameResolverShard shard) {
        if (!shard.isReady()) {
            return false;
        }

        if (now - shard.metrics().createdAtMs < opts.minUptimeToRunMs) {
            return false;
        }

        if (shard.metrics().resources.get() <= 0) {
            return false;
        }

        if (activeTask.containsKey(shard.cloudId)) {
            return false;
        }

        var status = statusByShardId.get(shard.cloudId);
        if (status == null) {
            status = new Status(now, randomPeriod(opts.firstRunPeriodMs));
            statusByShardId.put(shard.cloudId, status);
        }

        return now - status.createdAtMillis >= status.nextPeriod;
    }

    private CompletableFuture<Void> runTtlTask(NameResolverShard shard, Object2LongMap<String> metricTtlByService, long now) {
        logger.info("Starting TTL task for shard {}", shard.cloudId);
        CompletableFuture<Void> future;
        try {
            var validators = opts.lightValidations
                    ? ResourceValidator.LIGHT_VALIDATOR
                    : ResourceValidator.STRICT_VALIDATOR;
            var ttlPredicate = new TtlPredicate(now, opts.resourceTtlMs, metricTtlByService, validators);
            var predicate = deletePredicate.or(ttlPredicate);
            var task = new TtlTask(shard, issueTracker, predicate);
            activeTask.put(shard.cloudId, task);
            future = CompletableFuture.completedFuture(task)
                    .thenComposeAsync(TtlTask::run, executor);
        } catch (Throwable e) {
            future = CompletableFuture.failedFuture(e);
        }

        metrics.taskMetrics.forFuture(future);
        return future.whenComplete((ignore, e) -> {
            onCompleteTtlTask(shard, e);
            activeTask.remove(shard.cloudId);
            actor.schedule();
        });
    }

    private void onCompleteTtlTask(NameResolverShard shard, @Nullable Throwable e) {
        long now = System.currentTimeMillis();
        if (e != null) {
            shard.error(new RuntimeException("TTL task for shard " + shard.cloudId + " failed", e));
            var prev = statusByShardId.put(shard.cloudId, new Status(now, randomPeriod(opts.failRunPeriodMs)));
            if (prev != null) {
                executor.execute(() -> prev.nextFuture.completeExceptionally(e));
            }
        } else {
            var prev = statusByShardId.put(shard.cloudId, new Status(now, randomPeriod(opts.successRunPeriodMs)));
            if (prev != null) {
                executor.execute(() -> prev.nextFuture.complete(null));
            }
        }
    }

    private long randomPeriod(long millis) {
        long half = millis / 2;
        return half + ThreadLocalRandom.current().nextLong(half);
    }

    @Override
    public void close() {
        closed = true;
        pereodicallyAct.cancel(false);
    }

    @Override
    public void onUpdateServiceProviders(Map<String, ServiceProvider> serviceProviderById) {
        var ttlByService = new Object2LongOpenHashMap<String>(serviceProviderById.size());
        for (var provider : serviceProviderById.values()) {
            long ttlMillis = TimeUnit.DAYS.toMillis(provider.getShardSettings().getMetricsTtlDays());
            ttlByService.put(provider.getId(), ttlMillis);
        }
        this.metricsTtlByService = ttlByService;
    }

    public record Status(long createdAtMillis, long nextPeriod, CompletableFuture<Void> nextFuture) {
        public Status(long createdAtMillis, long nextPeriod) {
            this(createdAtMillis, nextPeriod, new CompletableFuture<>());
        }
    }

    private static class Metrics {
        final AsyncMetrics taskMetrics;

        public Metrics(MetricRegistry registry) {
            this.taskMetrics = new AsyncMetrics(registry, "nameResolver.ttlTask");
        }
    }
}
