package ru.yandex.solomon.name.resolver.ttl;

import java.util.concurrent.TimeUnit;

import ru.yandex.misc.lang.DefaultToString;

/**
 * @author Vladimir Gordiychuk
 */
public class TtlSchedulerOpts extends DefaultToString {
    public final long firstRunPeriodMs;
    public final long failRunPeriodMs;
    public final long successRunPeriodMs;
    public final long minUptimeToRunMs;
    public final long schedulePeriodMs;
    public final int maxInFlight;
    public final long resourceTtlMs;
    public final boolean lightValidations;

    private TtlSchedulerOpts(Builder builder) {
        this.firstRunPeriodMs = builder.firstRunPeriodMs != 0
                ? builder.firstRunPeriodMs
                : TimeUnit.HOURS.toMillis(3);
        this.failRunPeriodMs = builder.failRunPeriodMs != 0
                ? builder.failRunPeriodMs
                : TimeUnit.HOURS.toMillis(1);
        this.successRunPeriodMs = builder.successRunPeriodMs != 0
                ? builder.successRunPeriodMs
                : TimeUnit.HOURS.toMillis(24);
        this.minUptimeToRunMs = builder.minUptimeToRunMs != 0
                ? builder.minUptimeToRunMs
                : TimeUnit.HOURS.toMillis(1);
        this.schedulePeriodMs = builder.schedulePeriodMs != 0
                ? builder.schedulePeriodMs
                : TimeUnit.MINUTES.toMillis(10);
        this.maxInFlight = builder.maxInFlight != 0
                ? builder.maxInFlight
                : 50;
        this.resourceTtlMs = builder.resourceTtlMs != 0
                ? builder.resourceTtlMs
                : TimeUnit.DAYS.toMillis(3);
        this.lightValidations = builder.lightValidations;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private long firstRunPeriodMs;
        private long failRunPeriodMs;
        private long successRunPeriodMs;
        private long minUptimeToRunMs;
        private long schedulePeriodMs;
        private int maxInFlight;
        private long resourceTtlMs;
        private boolean lightValidations;

        public Builder firstRunPeriod(long value, TimeUnit unit) {
            this.firstRunPeriodMs = unit.toMillis(value);
            return this;
        }

        public Builder failRunPeriod(long value, TimeUnit unit) {
            this.failRunPeriodMs = unit.toMillis(value);
            return this;
        }

        public Builder successRunPeriod(long value, TimeUnit unit) {
            this.successRunPeriodMs = unit.toMillis(value);
            return this;
        }

        public Builder minUptimeToRun(long value, TimeUnit unit) {
            this.minUptimeToRunMs = unit.toMillis(value);
            return this;
        }

        public Builder schedulePeriod(long value, TimeUnit unit) {
            this.schedulePeriodMs = unit.toMillis(value);
            return this;
        }

        public Builder maxInFlight(int maxInFlight) {
            this.maxInFlight = maxInFlight;
            return this;
        }

        public Builder resourceTtl(long value, TimeUnit unit) {
            this.resourceTtlMs = unit.toMillis(value);
            return this;
        }

        public Builder lightValidations(boolean lightValidations) {
            this.lightValidations = lightValidations;
            return this;
        }

        public TtlSchedulerOpts build() {
            return new TtlSchedulerOpts(this);
        }
    }

}
