package ru.yandex.solomon.name.resolver.ttl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

import it.unimi.dsi.fastutil.objects.Object2LongMap;
import it.unimi.dsi.fastutil.objects.Object2LongOpenHashMap;

import ru.yandex.solomon.name.resolver.IssueTracker;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.stats.ResourceKey;

/**
 * @author Vladimir Gordiychuk
 */
public class TtlTask {
    private static final int MAX_BATCH_SIZE = 1000;

    public final NameResolverShard shard;
    private final Iterator<Resource> it;
    private final IssueTracker tracker;
    private final long reindexThreshold;
    private final Predicate<Resource> predicate;
    private final Object2LongMap<ResourceKey> obsoleteByKey = new Object2LongOpenHashMap<>();
    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();

    public TtlTask(NameResolverShard shard, IssueTracker tracker, Predicate<Resource> predicate) {
        this.shard = shard;
        this.tracker = tracker;
        this.reindexThreshold = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(3);
        this.it = shard.resourceIterator();
        this.predicate = predicate;
    }

    public CompletableFuture<Void> run() {
        continueDelete();
        return doneFuture;
    }

    private void continueDelete() {
        List<Resource> candidates = new ArrayList<>();
        while (it.hasNext()) {
            var resource = it.next();
            if (predicate.test(resource)) {
                candidates.add(resource);
                if (candidates.size() >= MAX_BATCH_SIZE) {
                    delete(candidates);
                    return;
                }
            } else {
                checkObsoleteReindex(resource);
            }
        }

        if (!candidates.isEmpty()) {
            delete(candidates);
            return;
        }

        shard.updateObsolete(obsoleteByKey);
        doneFuture.complete(null);
    }

    private void delete(List<Resource> resources) {
        shard.delete(resources).whenComplete((ignore, e) -> {
            if (e != null) {
                doneFuture.completeExceptionally(e);
                return;
            }

            continueDelete();
        });
    }

    private void checkObsoleteReindex(Resource resource) {
        if (resource.deletedAt != 0) {
            return;
        }

        if (resource.updatedAt > reindexThreshold || resource.reindexAt > reindexThreshold) {
            return;
        }

        tracker.reindexObsolete(shard.cloudId, resource);
        var key = ResourceKey.of(resource);
        obsoleteByKey.put(key, obsoleteByKey.getLong(key) + 1);
    }

}
