package ru.yandex.solomon.name.resolver.www;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
@Import({
        ManagerWriterContext.class,
})
public class NameResolverLocalShardsWww {
    @Autowired
    private HttpAuthenticator authenticator;
    @Autowired
    private InternalAuthorizer authorizer;
    @Autowired
    private ManagerWriterContext contect;
    @Autowired
    private NameResolverLocalShards shards;

    @Bean
    public RootLink nameResolverLocalShardsLink() {
        return new RootLink("/local-shards", "Local shards");
    }

    @RequestMapping(value = "/local-shards", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> stockpileLocalShards(
            @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
            @RequestParam(value = "limit", defaultValue = "100") int limit,
            ServerHttpRequest request)
    {
        return authorize(request)
                .thenApply(account -> localShardsImpl(sortBy, limit));
    }

    private CompletableFuture<Account> authorize(ServerHttpRequest request) {
        return authenticator.authenticate(request)
                .thenCompose(authSubject -> authorizer.authorize(authSubject));
    }

    private String localShardsImpl(int sortBy, int limit) {
        List<Record> records = this.shards.stream()
                .map(Record::of)
                .sorted(Comparator.comparing(r -> r.cloudId))
                .limit(limit)
                .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("NameResolver local shards", contect, columns, records, sortBy).genString();
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
                Column.of(
                        "CloudId",
                        r -> {
                            return new AHref(namedObjectLink(new NamedObjectId(NameResolverShard.class, r.cloudId)), r.cloudId);
                        },
                        Comparator.comparing(o -> o.cloudId)),
                Column.of("Uptime",
                        r -> DurationUtils.formatDurationMillis(r.uptimeMillis),
                        Comparator.comparingLong(o2 -> o2.uptimeMillis)
                ),
                Column.of("Resources",
                        r -> {
                            var url = UrlUtils.addParameter("/resources", "filterByCloudId", r.cloudId);
                            return new AHref(url, DataSize.shortString(r.resources));
                        },
                        Comparator.comparingLong(o2 -> o2.resources)
                ),
                Column.of("Errors",
                        r -> DataSize.shortString(r.errors),
                        Comparator.comparingLong(o2 -> o2.errors)
                ),
                Column.of("Last Error",
                        r -> {
                            if (r.lastErrorInstant == 0) {
                                return "none";
                            } else {
                                return Instant.ofEpochMilli(r.lastErrorInstant);
                            }
                        },
                        Comparator.comparingLong(o2 -> o2.lastErrorInstant)
                )
        );
    }

    private static class Record implements TableRecord {
        String cloudId;
        long uptimeMillis;
        long resources;
        long errors;
        long lastErrorInstant;

        public static Record of(NameResolverShard shard) {
            var r = new Record();
            r.cloudId = shard.cloudId;
            var metrics = shard.metrics();
            r.resources = metrics.resources.get();
            r.uptimeMillis = metrics.uptimeMillis();
            r.lastErrorInstant = metrics.lastErrorInstant;
            return r;
        }
    }
}
