package ru.yandex.solomon.name.resolver.www;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import com.google.common.base.Throwables;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.staffOnly.UrlUtils;
import ru.yandex.solomon.staffOnly.html.CssLine;
import ru.yandex.solomon.staffOnly.manager.ManagerWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourcesPage extends ManagerPageTemplate {
    private static final int RECORDS_LIMIT = 1_000;
    private static final String FILTER_BY_SELECTORS_PARAM = "filterBySelectors";
    private static final String FILTER_BY_CLOUD_PARAM = "filterByCloudId";
    private static final String FILTER_BY_MODE_PARAM = "filterByMode";
    private static final String SORT_BY_COLUMN_PARAM = "sortBy";
    private final String path;
    private final Map<String, String> params;

    private final NameResolverLocalShards shards;

    // Filters
    private String filterBySelectors;
    private String filterByCloud;
    private Mode filterByMode;

    // Sorts
    private String sortByColumn;
    private boolean sortAsAsc;

    private List<Column<Record>> columns;
    private List<Record> records;
    private String warning;

    ResourcesPage(String path, Map<String, String> params, NameResolverLocalShards shards) {
        super("Resources");
        this.path = path;
        this.params = params;
        this.shards = shards;

        parseParameters();
        prepareColumn();
        prepareRecords();
    }

    private void parseParameters() {
        this.filterByCloud = getParam(FILTER_BY_CLOUD_PARAM);
        this.filterBySelectors = getParam(FILTER_BY_SELECTORS_PARAM);
        {
            String str = getParam(FILTER_BY_MODE_PARAM);
            this.filterByMode = str.isEmpty() ? null : Mode.valueOf(str.toUpperCase());
        }
        {
            String sortByStr = getParam(SORT_BY_COLUMN_PARAM);
            this.sortAsAsc = !sortByStr.startsWith("-");
            this.sortByColumn = sortByStr.startsWith("-") ? sortByStr.substring(1) : sortByStr;
        }
    }

    private String getParam(String name) {
        return params.getOrDefault(name, "").trim();
    }

    @Nullable
    private Predicate<Record> prepareFilterPredicate() {
        switch (Nullables.orDefault(filterByMode, Mode.ALL)) {
            case DELETED:
                return record -> record.resource.deletedAt != 0;
            case NON_DELETED:
                return record -> record.resource.deletedAt == 0;
            case NON_REPLACED:
                return record -> !record.resource.replaced;
            default:
                return null;
        }
    }

    @Nullable
    private Comparator<Record> prepareSorting() {
        if (sortByColumn.isEmpty()) {
            return null;
        }

        var sortBy = columns.stream()
                .filter(column -> column.title.equals(sortByColumn))
                .findFirst()
                .orElse(null);

        if (sortBy == null) {
            return null;
        }

        if (sortAsAsc) {
            return sortBy.comparator;
        } else {
            return sortBy.comparator.reversed();
        }
    }

    @Override
    protected void content() {
        tag("div.row", () -> {
            tag("div.well", this::writeFiltersForm);
        });
        writeTable();
    }

    private void writeFiltersForm() {
        form(() -> {
            tag("div.form-group", () -> {
                h3("Filters");
                table(() -> {
                    textInput(
                            FILTER_BY_CLOUD_PARAM,
                            "CloudId:",
                            filterByCloud);
                    textInput(
                            FILTER_BY_SELECTORS_PARAM,
                            "Selectors:",
                            filterBySelectors);
                    select(
                            FILTER_BY_MODE_PARAM,
                            "Mode:",
                            Stream.of(Mode.values()).map(Enum::name).collect(Collectors.toList()),
                            filterByMode == null ? " " : filterByMode.name());
                });
            });
            tag("div.form-group", () -> {
                buttonSubmitDefault("Apply");
                write(" ");
                tag("a.btn.btn-warning", new Attr("href", path), () -> write("Clear Filters"));
            });
        }, new Attr("action", path), new Attr("class", "form-horizontal"));
    }

    private void textInput(String id, String name, String value) {
        tr(() -> {
            thText(name);
            td(() -> {
                tag("input",
                        new Attr("id", id),
                        new Attr("type", "textfield"),
                        new Attr("name", id),
                        new Attr("value", value),
                        new Attr("class", "form-control"),
                        new Attr("size", "200"));
            });
        });
    }

    private void select(String id, String name, List<String> values, String defaultValue) {
        tr(() -> {
            thText(name);
            td(() -> {
                selectFormControl(id, id, values, defaultValue);
            });
        });
    }

    private void writeTable() {
        var writer = new ManagerWriter(new ManagerWriterContext(), this);
        if (!Strings.isNullOrEmpty(warning)) {
            div(() -> preText(warning));
        }
        div(() -> write("Records: " + records.size()));
        if (records.size() > RECORDS_LIMIT) {
            div(() -> write("Table truncated, add one more filter"));
        }
        tag("table", () -> {
            tr(() -> {
                for (var column : columns) {
                    th(() -> {
                        String title = column.title;
                        if (!title.equals(sortByColumn)) {
                            aHref(hrefWithParam(SORT_BY_COLUMN_PARAM, title), title);
                        } else {
                            String link = sortAsAsc ? "-" + title : title;
                            String icon = sortAsAsc
                                    ? "glyphicon-sort-by-attributes"
                                    : "glyphicon-sort-by-attributes-alt";
                            aHref(hrefWithParam(SORT_BY_COLUMN_PARAM, link), () -> {
                                write(column.title);
                                tag("span", Attr.cssClass("glyphicon " + icon));
                            });
                        }
                    });
                }
            });
            int recordCount = 0;
            for (var record : records) {
                if (recordCount++ >= RECORDS_LIMIT) {
                    break;
                }
                tr(() -> {
                    for (var column : columns) {
                        writer.writeCellTd(column.content.apply(record));
                    }
                });
            }
        }, Attr.cssClass("table simple-table2 table-hover table-condensed"),
                Attr.style(
                        new CssLine("word-break", "break-all")));
    }

    private String hrefWithParam(String name, String value) {
        return UrlUtils.hrefWithParam(name, value, path, params);
    }

    private void prepareRecords() {
        var shard = shards.getShardById(filterByCloud);
        if (shard == null) {
            warning = "Shard by id `" + filterByCloud+ "` not found on host";
            records = List.of();
            return;
        }

        try {
            var selector = Selectors.parse(filterBySelectors);
            var stream = shard.search(selector).map(Record::of);
            var filter = prepareFilterPredicate();
            if (filter != null) {
                stream = stream.filter(filter);
            }

            var comparator = prepareSorting();
            if (comparator != null) {
                stream = stream.sorted(comparator);
            }
            this.records = stream.collect(Collectors.toList());
        } catch (Throwable e) {
            warning = Throwables.getStackTraceAsString(e);
            this.records = List.of();
        }
    }

    private void prepareColumn() {
        List<Column<Record>> result = new ArrayList<>();
        result.add(Column.of(
                "folderId",
                r -> r.resource.folderId,
                Comparator.comparing(o -> o.resource.folderId)));
        result.add(Column.of(
                "service",
                r -> r.resource.service,
                Comparator.comparing(o -> o.resource.service)));
        result.add(Column.of(
                "type",
                r -> r.resource.type,
                Comparator.comparing(o -> o.resource.type)));
        result.add(Column.of(
                "resourceId",
                r -> r.resource.resourceId,
                Comparator.comparing(o -> o.resource.resourceId)));
        result.add(Column.of(
                "name",
                r -> r.resource.name,
                Comparator.comparing(o -> o.resource.name)));
        result.add(Column.of(
                "updatedAt",
                r -> Instant.ofEpochMilli(r.resource.updatedAt),
                Comparator.comparingLong(o -> o.resource.updatedAt)));
        result.add(Column.of(
                "deletedAt",
                r -> r.resource.deletedAt > 0 ? Instant.ofEpochMilli(r.resource.deletedAt).toString() : "",
                Comparator.comparingLong(o -> o.resource.deletedAt)));
        result.add(Column.of(
                "reindexAt",
                r -> r.resource.reindexAt > 0 ? Instant.ofEpochMilli(r.resource.reindexAt).toString() : "",
                Comparator.comparingLong(o -> o.resource.reindexAt)));
        result.add(Column.of(
                "replaced",
                r -> r.resource.replaced,
                (o1, o2) -> Boolean.compare(o1.resource.replaced, o2.resource.replaced)));
        this.columns = result;
    }

    private static class Record implements TableRecord {
        Resource resource;
        public static Record of(Resource resource) {
            var r = new Record();
            r.resource = resource;
            return r;
        }
    }

    private enum Mode {
        ALL, DELETED, NON_DELETED, NON_REPLACED
    }
}
