package ru.yandex.solomon.name.resolver;

import java.util.Comparator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.name.resolver.db.InMemoryResourceDao;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverLocalShardsTest {
    private NameResolverLocalShards shards;
    private NameResolverShardFactoryStub stub;

    @Before
    public void setUp() {
        shards = new NameResolverLocalShards();
        stub = new NameResolverShardFactoryStub();
    }

    @After
    public void tearDown() {
        stub.close();
    }

    @Test
    public void getNoExist() {
        assertNull(shards.getShardById("notExist"));
        assertNull(shards.getShardById("solomon"));
    }

    @Test
    public void emptyStream() {
        var result = shards.stream().collect(Collectors.toList());
        assertEquals(List.of(), result);
    }

    @Test
    public void emptyIterator() {
        var it = shards.iterator();
        assertFalse(it.hasNext());

        try {
            var shard = it.next();
            fail("next: " + shard);
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void addShard() {
        var shard = newShard("alice");
        assertTrue(shards.addShard(shard));
        assertFalse(shards.addShard(shard));

        assertSame(shard, shards.getShardById("alice"));

        {
            var it = shards.iterator();
            assertTrue(it.hasNext());
            assertSame(shard, it.next());
            assertFalse(it.hasNext());
        }

        {
            var list = shards.stream().collect(Collectors.toList());
            assertEquals(1, list.size());
            assertSame(shard, list.get(0));
        }
    }

    @Test
    public void addMultipleShard() {
        var alice = newShard("alice");
        var bob = newShard("bob");

        assertTrue(shards.addShard(alice));
        assertTrue(shards.addShard(bob));

        assertSame(alice, shards.getShardById("alice"));
        assertSame(bob, shards.getShardById("bob"));

        {
            var list = Lists.newArrayList(shards.iterator());
            list.sort(Comparator.comparing(o -> o.cloudId));
            assertSame(alice, list.get(0));
            assertSame(bob, list.get(1));
            assertEquals(2, list.size());
        }

        {
            var list = shards.stream().collect(Collectors.toList());
            list.sort(Comparator.comparing(o -> o.cloudId));
            assertSame(alice, list.get(0));
            assertSame(bob, list.get(1));
            assertEquals(2, list.size());
        }
    }

    @Test
    public void removeShard() {
        var alice = newShard("alice");
        var bob = newShard("bob");

        assertTrue(shards.addShard(alice));
        assertTrue(shards.addShard(bob));

        assertTrue(shards.remove(alice));
        assertFalse(shards.remove(alice));

        assertNull(shards.getShardById("alice"));
        assertSame(bob, shards.getShardById("bob"));
    }

    @Test
    public void gracefulShutdown() {
        var alice = newShard("alice");
        var bob = newShard("bob");

        assertTrue(shards.addShard(alice));
        assertTrue(shards.addShard(bob));
        assertNotEquals(ShardState.CLOSED, alice.getState());
        assertNotEquals(ShardState.CLOSED, bob.getState());

        shards.gracefulShutdown().join();

        assertEquals(ShardState.CLOSED, alice.getState());
        assertEquals(ShardState.CLOSED, bob.getState());
    }

    public NameResolverShard newShard(String id) {
        var executor = ForkJoinPool.commonPool();
        var dao = new InMemoryResourceDao();
        var seqNo = new AssignmentSeqNo(id.hashCode(), id.hashCode());
        return stub.create(id, seqNo);
    }
}
