package ru.yandex.solomon.name.resolver;

import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import io.grpc.Status;
import io.grpc.Status.Code;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.stats.ReceiveMetrics;
import ru.yandex.solomon.name.resolver.stats.ResourceKey;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverShardTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private NameResolverShardFactoryStub stub;
    private NameResolverShard shard;

    @Before
    public void setUp() {
        stub = new NameResolverShardFactoryStub();
        createShard();
    }

    @After
    public void tearDown() {
        stub.close();
        shard.close();
    }

    @Test
    public void addResource() {
        var expect = resource();
        update(expect).join();
        assertEquals(List.of(expect), stub.resourceDao.findResources(expect.cloudId).join());
    }

    @Test
    public void updateResource() {
        var v1 = resource();
        update(v1).join();

        var v2 = new Resource(v1)
                .setName("changed-name-in-test")
                .setUpdatedAt(v1.updatedAt + 1_000);
        update(v2).join();
        assertEquals(List.of(v2), stub.resourceDao.findResources(v2.cloudId).join());
    }

    @Test
    public void obsoleteResource() {
        var v1 = resource();
        update(v1).join();

        var v2 = new Resource(v1)
                .setName("obsolete-name")
                .setUpdatedAt(v1.updatedAt + 1_000);

        var v3 = new Resource(v2)
                .setName("actual-resource-name")
                .setUpdatedAt(v2.updatedAt + 1_000);

        update(v3).join();
        update(v2).join(); // should be ignored
        assertEquals(List.of(v3), stub.resourceDao.findResources(v3.cloudId).join());
    }

    @Test
    public void obsoleteTogetherWithActual() {
        var v1 = resource();

        var v2 = new Resource(v1)
                .setName("obsolete-name")
                .setUpdatedAt(v1.updatedAt + 1_000);

        var v3 = new Resource(v2)
                .setName("actual-resource-name")
                .setUpdatedAt(v2.updatedAt + 1_000);

        CompletableFuture.allOf(update(v1), update(v3), update(v2)).join();
        assertEquals(List.of(v3), stub.resourceDao.findResources(v3.cloudId).join());
        assertEquals(List.of(v3), shard.search(Selectors.parse("resource=actual-resource-name")).collect(Collectors.toList()));
    }

    @Test
    public void obsoleteUpdateDeleted() {
        var v1 = resource();
        update(v1).join();
        assertEquals(List.of(v1), stub.resourceDao.findResources(v1.cloudId).join());

        var v2 = new Resource(v1)
                .setName("obsolte-name-update")
                .setUpdatedAt(v1.updatedAt + 1_000);

        var v3 = new Resource(v1)
                .setUpdatedAt(v2.updatedAt + 1_000)
                .setDeletedAt(v2.updatedAt + 1_000);

        update(v3).join();
        update(v2).join(); // should be ignored
        assertEquals(List.of(v3), stub.resourceDao.findResources(v3.cloudId).join());
    }

    @Test
    public void markResourceDelete() {
        var v1 = resource();
        update(v1).join();

        var v2 = new Resource(v1)
                .setUpdatedAt(v1.updatedAt + 1_000)
                .setDeletedAt(System.currentTimeMillis())
                .setName("");
        update(v2).join();

        var expected = new Resource(v2).setName(v1.name);
        assertEquals(List.of(expected), stub.resourceDao.findResources(v2.cloudId).join());
    }

    @Test
    public void markResourceDeleteAfterUpdate() {
        var v1 = resource();
        update(v1).join();

        update(resource().setResourceId("123").setService("service2")).join();

        var v2 = new Resource(v1)
                .setResourceId("resourceId123")
                .setUpdatedAt(v1.updatedAt + 1_000)
                .setName("123");
        shard.update(List.of(v2), true, "service").join();

        final List<Resource> join = stub.resourceDao.findResources(v2.cloudId).join();
        assertEquals(List.of(resource().setResourceId("123").setService("service2"), v2), join);
    }


    @Test
    public void deleteResource() {
        var alice = resource().setResourceId("alice").setName("alice");
        var bob = resource().setResourceId("bob").setName("bob");

        update(alice, bob).join();

        assertEquals(Set.of(alice, bob), Set.copyOf(stub.resourceDao.findResources(shard.cloudId).join()));
        assertEquals(List.of(alice), search("resource=alice"));
        assertEquals(List.of(bob), search("resource=bob"));

        delete(alice).join();
        assertEquals(List.of(bob), stub.resourceDao.findResources(shard.cloudId).join());
        assertEquals(List.of(), search("resource=alice"));
        assertEquals(List.of(bob), search("resource=bob"));
    }

    @Test
    public void multipleUpdateSameResource() {
        var v1 = resource();
        update(v1).join();

        IntStream.rangeClosed(0, 100_000)
                .mapToObj(idx -> new Resource(v1).setUpdatedAt(v1.updatedAt + idx).setName(Integer.toString(idx)))
                .map(this::update)
                .collect(collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid))
                .join();

        var expected = new Resource(v1)
                .setUpdatedAt(v1.updatedAt + 100_000)
                .setName(Integer.toString(100_000));

        assertEquals(List.of(expected), stub.resourceDao.findResources(expected.cloudId).join());
    }

    @Test
    public void shardAtStartupLoadResources() {
        var v1 = resource();
        update(v1).join();

        var v2 = new Resource(v1)
                .setName("obsolete-name")
                .setUpdatedAt(v1.updatedAt + 1_000);

        var v3 = new Resource(v2)
                .setName("actual-resource-name")
                .setUpdatedAt(v2.updatedAt + 1_000);

        update(v3).join();

        createShard();
        update(v2).join(); // should be ignored
        assertEquals(List.of(v3), stub.resourceDao.findResources(v3.cloudId).join());
    }

    @Test
    public void cancelUpdatesOnCloseShard() {
        var v1 = resource();
        update(v1).join();

        ForkJoinPool.commonPool().submit(() -> shard.close());
        Status status = IntStream.rangeClosed(0, 100_000)
                .mapToObj(idx -> new Resource(v1).setUpdatedAt(v1.updatedAt + idx).setName(Integer.toString(idx)))
                .map(this::update)
                .collect(collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid))
                .thenApply(ignore -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertEquals(status.toString(), Code.CANCELLED, status.getCode());
    }

    @Test
    public void cancelDeletesOnCloseShard() {
        var init = resource();
        var resources = IntStream.rangeClosed(0, 100_000)
                .mapToObj(idx -> new Resource(init).setResourceId(Integer.toString(idx)).setName("name-"+Integer.toString(idx)))
                .toArray(Resource[]::new);

        update(resources).join();

        ForkJoinPool.commonPool().submit(() -> shard.close());
        Status status = Stream.of(resources)
                .map(this::delete)
                .collect(collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid))
                .thenApply(ignore -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertEquals(status.toString(), Code.CANCELLED, status.getCode());
    }

    @Test
    public void daoUnavailableOnInit() throws InterruptedException {
        stub.resourceDao.setError(Status.UNAVAILABLE.withDescription("test").asRuntimeException());
        createShard();

        var resource = resource();
        CountDownLatch sync = new CountDownLatch(1);
        update(resource).whenComplete((ignore, e) -> sync.countDown());

        assertFalse(sync.await(100L, TimeUnit.MILLISECONDS));
        stub.resourceDao.setError(null);
        do {
            stub.clock.passedTime(1, TimeUnit.SECONDS);
        } while (!sync.await(1, TimeUnit.MILLISECONDS));
        assertEquals(List.of(resource), stub.resourceDao.findResources(resource.cloudId).join());
    }

    @Test
    public void daoUnavailableOnUpdate() throws InterruptedException {
        var v1 = resource();
        update(v1).join();

        var v2 = new Resource(v1)
                .setName("changed-name-in-test")
                .setUpdatedAt(v1.updatedAt + 1_000);

        CountDownLatch sync = new CountDownLatch(1);
        stub.resourceDao.setError(Status.UNAVAILABLE.withDescription("test").asRuntimeException());
        update(v2).whenComplete((ignore, e) -> sync.countDown());

        assertFalse(sync.await(100L, TimeUnit.MILLISECONDS));
        stub.resourceDao.setError(null);
        do {
            stub.clock.passedTime(1, TimeUnit.SECONDS);
        } while (!sync.await(1, TimeUnit.MILLISECONDS));
        assertEquals(List.of(v2), stub.resourceDao.findResources(v2.cloudId).join());
    }

    @Test
    public void daoUnavailableOnDelete() throws InterruptedException {
        var v1 = resource();
        update(v1).join();

        CountDownLatch sync = new CountDownLatch(1);
        stub.resourceDao.setError(Status.UNAVAILABLE.withDescription("test").asRuntimeException());
        delete(v1).whenComplete((ignore, e) -> sync.countDown());

        assertFalse(sync.await(100L, TimeUnit.MILLISECONDS));
        stub.resourceDao.setError(null);
        do {
            stub.clock.passedTime(1, TimeUnit.SECONDS);
        } while (!sync.await(1, TimeUnit.MILLISECONDS));
        assertEquals(List.of(), stub.resourceDao.findResources(v1.cloudId).join());
    }

    @Test
    public void searchResourceByName() {
        var alice = resource().setResourceId("resource-01").setName("alice");
        var bob = resource().setResourceId("resource-02").setName("bob");
        update(alice).join();
        update(bob).join();

        {
            var resources = shard.search(SelectorsFormat.parse("resource=alice")).collect(Collectors.toList());
            assertEquals(List.of(alice), resources);
        }
        {
            var resources = shard.search(SelectorsFormat.parse("resource=a*")).collect(Collectors.toList());
            assertEquals(List.of(alice), resources);
        }
        {
            var resources = shard.search(SelectorsFormat.parse("resource=bob")).collect(Collectors.toList());
            assertEquals(List.of(bob), resources);
        }
    }

    @Test
    public void resolveResourceById() {
        var alice = resource().setResourceId("resource-01").setName("alice");
        var bob = resource().setResourceId("resource-02").setName("bob");
        update(alice).join();
        update(bob).join();

        assertEquals(List.of(alice, bob), shard.resolve(List.of("resource-01", "resource-02")));
        assertEquals(List.of(alice), shard.resolve(List.of("resource-01", "not_exists")));
        assertEquals(List.of(bob), shard.resolve(List.of("resource-01-not_exists", "resource-02")));
    }

    @Test
    public void searchByIdExactOne() {
        var alice = resource().setResourceId("resource-01").setType("vm").setName("alice");
        var bob = resource().setResourceId("resource-02").setType("vm").setName("bob");
        update(alice, bob).join();

        assertEquals(List.of(alice), search("resource=resource-01"));
        assertEquals(List.of(alice), search("resource==resource-01"));
        assertEquals(List.of(alice), search("resource=resource-01, type=vm"));
        assertEquals(List.of(alice), search("resource==resource-01, type=vm"));
        assertEquals(List.of(alice), search("resource==resource-01, type!=disk"));
        assertEquals(List.of(), search("resource==resource-01, type=disk"));
        assertEquals(List.of(), search("resource==resource-01, resource==resource-02"));
        assertEquals(List.of(), search("resource=resource-01, resource=resource-02"));
        assertEquals(List.of(), search("resource==resource-01, folderId=not_exist_folder"));
        assertEquals(List.of(), search("resource==resource-01, service=not_exist_service"));
    }

    @Test
    public void searchByNameExactMany() {
        var alice = resource().setResourceId("resource-01").setType("vm").setName("alice");
        var bob = resource().setResourceId("resource-02").setType("vm").setName("bob");
        var eva = resource().setResourceId("resource-03").setType("vm").setName("eva");
        update(alice, bob, eva).join();

        assertEquals(Set.of(alice, bob), Set.copyOf(search("resource='alice|bob'")));
        assertEquals(Set.of(alice, bob), Set.copyOf(search("resource='alice|bob', type=vm")));
        assertEquals(Set.of(bob, eva), Set.copyOf(search("resource='bob|eva'")));
        assertEquals(Set.of(bob, eva), Set.copyOf(search("resource='alice|bob|eva', resource='bob|eva'")));
        assertEquals(Set.of(bob, eva), Set.copyOf(search("resource='bob|eva', resource='alice|bob|eva'")));
        assertEquals(Set.of(bob, eva), Set.copyOf(search("resource='bob|eva', resource='alice|bob|eva', type=vm")));

        assertEquals(List.of(), search("resource='alice|bob', type=disk"));
        assertEquals(List.of(), search("resource='alice|bob', resource==resource-10"));
        assertEquals(List.of(), search("resource='alice|bob', resource=resource-10"));
        assertEquals(List.of(), search("resource='alice|bob', folderId=not_exist_folder"));
        assertEquals(List.of(), search("resource='alice|bob', service=not_exist_service"));
    }

    @Test
    public void searchByNotName() {
        var alice = resource().setResourceId("resource-01").setType("vm").setName("alice");
        var bob = resource().setResourceId("resource-02").setType("vm").setName("bob");
        update(alice, bob).join();

        assertEquals(List.of(bob), search("resource!=alice"));
        assertEquals(List.of(bob), search("resource!==alice"));
        assertEquals(List.of(bob), search("resource!=a*"));
        assertEquals(List.of(bob), search("resource!=alice|eva"));
        assertEquals(List.of(alice), search("resource='*', resource!=bob"));
    }

    @Test
    public void reindexLostAdd() {
        var alice = resource().setName("alice").setResourceId("one").setReindexAt(System.currentTimeMillis());
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        assertEquals(List.of(alice), search("name=alice"));
        assertEquals(1L, receiveMetrics(alice).receiveLost.get());
        assertEquals(1L, receiveMetrics(bob).receiveAdd.get());
    }

    @Test
    public void reindexLostUpdate() {
        var now = System.currentTimeMillis();
        var alice = resource().setName("alice").setResourceId("one").setUpdatedAt(now);
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        var aliceV2 = new Resource(alice).setName("aliceV2").setUpdatedAt(now + 1_000).setReindexAt(now + 10_000);
        update(aliceV2).join();

        assertEquals(List.of(aliceV2), search("name=a*"));
        assertEquals(1L, receiveMetrics(alice).receiveLost.get());
        assertEquals(2L, receiveMetrics(bob).receiveAdd.get());
    }

    @Test
    public void reindexLostDelete() {
        var now = System.currentTimeMillis();
        var alice = resource().setName("alice").setResourceId("one").setUpdatedAt(now);
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        var aliceV2 = new Resource(alice).setName("aliceV2")
                .setUpdatedAt(now + 1_000)
                .setDeletedAt(now + 1_000)
                .setReindexAt(now + 10_000);
        update(aliceV2).join();

        assertEquals(List.of(aliceV2), search("name=a*"));
        assertEquals(1L, receiveMetrics(alice).receiveLost.get());
        assertEquals(2L, receiveMetrics(bob).receiveAdd.get());
    }

    @Test
    public void reindexSame() {
        var now = System.currentTimeMillis();
        var alice = resource().setName("alice").setResourceId("one").setUpdatedAt(now);
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        var aliceV2 = new Resource(alice).setReindexAt(now + 1_000);
        update(aliceV2).join();

        assertEquals(List.of(aliceV2), search("name=a*"));
        assertEquals(0L, receiveMetrics(alice).receiveLost.get());
        assertEquals(2L, receiveMetrics(bob).receiveAdd.get());
        assertEquals(1L, receiveMetrics(bob).receiveReindex.get());
    }

    @Test
    public void reindexTsEqualToUpdateTs() {
        var now = System.currentTimeMillis();
        var alice = resource().setName("alice").setResourceId("one").setUpdatedAt(now);
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        // Not all service providers save timestamp for update and send updatedAt as updateAt=reindexTs
        var aliceV2 = new Resource(alice).setUpdatedAt(now + 1_000).setReindexAt(now + 1_000);
        update(aliceV2).join();

        var expected = new Resource(alice).setReindexAt(now + 1_000);

        assertEquals(List.of(expected), search("name=a*"));
        assertEquals(0L, receiveMetrics(alice).receiveLost.get());
        assertEquals(2L, receiveMetrics(bob).receiveAdd.get());
        assertEquals(1L, receiveMetrics(bob).receiveReindex.get());
    }

    @Test
    public void reindexTsEqualToDeleteTs() {
        var now = System.currentTimeMillis();
        var alice = resource().setName("alice").setResourceId("one").setUpdatedAt(now).setDeletedAt(now);
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        // Not all service providers save timestamp for update and send updatedAt as updateAt=reindexTs
        var aliceV2 = new Resource(alice)
                .setUpdatedAt(now + 1_000)
                .setDeletedAt(now + 1_000)
                .setReindexAt(now + 1_000);
        update(aliceV2).join();

        var expected = new Resource(alice).setReindexAt(now + 1_000);

        assertEquals(List.of(expected), search("name=a*"));
        assertEquals(0L, receiveMetrics(alice).receiveLost.get());
        assertEquals(1L, receiveMetrics(bob).receiveAdd.get());
        assertEquals(1L, receiveMetrics(bob).receiveDelete.get());
        assertEquals(1L, receiveMetrics(bob).receiveReindex.get());
    }

    @Test
    public void reindexTsEqualToDeleteTsLostDelete() {
        var now = System.currentTimeMillis();
        var alice = resource().setName("alice").setResourceId("one").setUpdatedAt(now);
        var bob = resource().setName("bob").setResourceId("two");
        update(alice, bob).join();

        // Not all service providers save timestamp for update and send updatedAt as updateAt=reindexTs
        var aliceV2 = new Resource(alice)
                .setUpdatedAt(now + 1_000)
                .setDeletedAt(now + 1_000)
                .setReindexAt(now + 1_000);
        update(aliceV2).join();

        assertEquals(List.of(aliceV2), search("name=a*"));
        assertEquals(1L, receiveMetrics(alice).receiveLost.get());
        assertEquals(2L, receiveMetrics(bob).receiveAdd.get());
    }

    @Test
    public void updateAfterReindex() {
        update(resource().setUpdatedAt(millis("2021-07-04T19:46:18.767Z")).setReindexAt(millis("2021-07-15T23:19:52.648Z"))).join();
        update(resource().setUpdatedAt(millis("2021-07-04T19:46:18.767Z")).setReindexAt(millis("2021-07-16T23:20:18.772Z"))).join();
        update(resource().setUpdatedAt(millis("2021-07-17T15:46:43.051Z")).setReindexAt(0L)).join();
        update(resource().setUpdatedAt(millis("2021-07-17T16:16:44.752Z")).setReindexAt(0L)).join();
        update(resource().setUpdatedAt(millis("2021-07-17T16:21:44.849Z")).setReindexAt(0L)).join();

        var expect = resource().setUpdatedAt(millis("2021-07-17T16:21:44.849Z"));
        var actual = search("");
        assertEquals(List.of(expect), actual);

        var metrics = receiveMetrics(expect);
        assertEquals(1L, metrics.receiveLost.get());
        assertEquals(1L, metrics.receiveReindex.get());
        assertEquals(3L, metrics.receiveUpdate.get());
        assertEquals(0L, metrics.receiveObsolete.get());
    }

    private static long millis(String time) {
        return Instant.parse(time).toEpochMilli();
    }

    private ReceiveMetrics receiveMetrics(Resource resource) {
        return stub.metrics.receive.get(ResourceKey.of(resource));
    }

    private void createShard() {
        if (shard != null) {
            shard.close();
        }

        var seqNo = new AssignmentSeqNo(1, 2);
        shard = stub.create("myCloudId", seqNo);
        shard.start();
    }

    private Resource resource() {
        return staticResource().setCloudId("myCloudId");
    }

    private List<Resource> search(String selectors) {
        return shard.search(Selectors.parse(selectors))
                .collect(Collectors.toList());
    }

    private CompletableFuture<Void> update(Resource... resources) {
        return Stream.of(resources)
                .map(shard::update)
                .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid));
    }

    private CompletableFuture<Void> delete(Resource... resources) {
        return shard.delete(Arrays.asList(resources));
    }
}
