package ru.yandex.solomon.name.resolver;

import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import io.grpc.Status;
import io.grpc.Status.Code;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.ResolveRequest;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceServiceImplTest {
    private NameResolverLocalShards shards;
    private NameResolverShardFactoryStub shardFactory;
    private ResourceService service;

    @Before
    public void setUp() throws Exception {
        shards = new NameResolverLocalShards();
        shardFactory = new NameResolverShardFactoryStub();

        for (var shardId : List.of("one", "two")) {
            var shard = shardFactory.create(shardId, new AssignmentSeqNo(1, shardId.hashCode()));
            shards.addShard(shard);
            shard.start().join();
        }

        service = new ResourceServiceImpl(shards);
    }

    @After
    public void tearDown() {
        shards.gracefulShutdown().join();
        shardFactory.close();
    }

    @Test
    public void findAbsentShard() {
        var status = service.find(FindRequest.newBuilder()
                .cloudId("not_exist")
                .build())
                .thenApply(response -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertEquals(status.toString(), Code.NOT_FOUND, status.getCode());
    }

    @Test
    public void findEmpty() {
        var result = service.find(FindRequest.newBuilder()
                .cloudId("one")
                .selectors(Selectors.parse("resource=my-resource"))
                .build())
                .join();

        assertEquals(List.of(), result.resources);
        assertFalse(result.truncated);
    }

    @Test
    public void find() {
        var alice = staticResource().setCloudId("one").setResourceId("r-1").setName("alice");
        var bob = staticResource().setCloudId("one").setResourceId("r-2").setName("bob");
        var aliceTwo = staticResource().setCloudId("two").setResourceId("r-3").setName("alice");
        service.updateResources("one", List.of(alice, bob), false, "").join();
        service.updateResources("two", List.of(aliceTwo), false, "").join();

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=alice"))
                    .build())
                    .join();

            assertEquals(List.of(alice), response.resources);
            assertFalse(response.truncated);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("two")
                    .selectors(Selectors.parse("resource=alice"))
                    .build())
                    .join();

            assertEquals(List.of(aliceTwo), response.resources);
            assertFalse(response.truncated);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=alice|bob"))
                    .limit(2)
                    .build())
                    .join();

            assertEquals(Set.of(alice, bob), Set.copyOf(response.resources));
            assertFalse(response.truncated);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=alice|bob"))
                    .limit(1)
                    .build())
                    .join();

            assertTrue(response.truncated);
            assertTrue(response.resources.toString(), response.resources.contains(alice) || response.resources.contains(bob));
        }
    }

    @Test
    public void findFilterReplaced() {
        var now = System.currentTimeMillis();
        var aliceReplaced = staticResource().setCloudId("one").setResourceId("r-1").setName("alice").setUpdatedAt(now);
        var alice = staticResource().setCloudId("one").setResourceId("r-2").setName("alice").setUpdatedAt(now + 1);
        service.updateResources("one", List.of(aliceReplaced, alice), false, "").join();

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=alice"))
                    .build())
                    .join();

            assertEquals(Set.of(alice, new Resource(aliceReplaced).setReplaced(true)), Set.copyOf(response.resources));
            assertFalse(response.truncated);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=alice"))
                    .filterReplaced(true)
                    .build())
                    .join();

            assertEquals(List.of(alice), response.resources);
            assertFalse(response.truncated);
        }
    }

    @Test
    public void resolveAbsentShard() {
        var status = service.resolve(ResolveRequest.newBuilder()
                .cloudId("not_exist")
                .resourceIds(List.of("one", "two"))
                .build())
                .thenApply(response -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertEquals(status.toString(), Code.NOT_FOUND, status.getCode());
    }

    @Test
    public void resolveEmpty() {
        var result = service.resolve(ResolveRequest.newBuilder()
                .cloudId("one")
                .resourceIds(List.of("one", "two"))
                .build())
                .join();

        assertEquals(List.of(), result.resources);
    }

    @Test
    public void resolve() {
        var alice = staticResource().setCloudId("one").setResourceId("r-1").setName("alice");
        var bob = staticResource().setCloudId("one").setResourceId("r-2").setName("bob");
        var aliceTwo = staticResource().setCloudId("two").setResourceId("r-3").setName("alice");
        service.updateResources("one", List.of(alice, bob), false, "").join();
        service.updateResources("two", List.of(aliceTwo), false, "").join();

        {
            var response = service.resolve(ResolveRequest.newBuilder()
                    .cloudId("one")
                    .resourceIds(List.of("r-1"))
                    .build())
                    .join();

            assertEquals(List.of(alice), response.resources);
        }

        {
            var response = service.resolve(ResolveRequest.newBuilder()
                    .cloudId("two")
                    .resourceIds(List.of("r-3"))
                    .build())
                    .join();

            assertEquals(List.of(aliceTwo), response.resources);
        }

        {
            var response = service.resolve(ResolveRequest.newBuilder()
                    .cloudId("one")
                    .resourceIds(List.of("r-1", "r-2"))
                    .build())
                    .join();

            assertEquals(List.of(alice, bob), response.resources);
        }

        {
            var response = service.resolve(ResolveRequest.newBuilder()
                    .cloudId("one")
                    .resourceIds(List.of("r-1", "r-5"))
                    .build())
                    .join();

            assertEquals(List.of(alice), response.resources);
        }
    }

    @Test
    public void updateRemove() {
        long now = System.currentTimeMillis();
        long deadline = now - TimeUnit.HOURS.toMillis(2);
        var alice = staticResource().setCloudId("one").setResourceId("r-1").setName("alice").setUpdatedAt(deadline - 10_0000);
        var bob = staticResource().setCloudId("one").setResourceId("r-2").setName("bob").setUpdatedAt(deadline - 10_0000);
        var bob2 = staticResource().setCloudId("one").setResourceId("r-3").setName("bob2").setUpdatedAt(deadline + 10_0000);
        var bob3 = staticResource().setCloudId("one").setResourceId("r-4").setName("bob3").setService("service2").setUpdatedAt(deadline - 10_0000);
        service.updateResources("one", List.of(alice, bob, bob2, bob3), false, "").join();
        service.updateResources("one", List.of(alice), true, "service").join();

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=alice"))
                    .build())
                    .join();

            assertEquals(List.of(alice), response.resources);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=bob"))
                    .build())
                    .join();

            assertEquals(List.of(), response.resources);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=bob3"))
                    .build())
                    .join();

            assertEquals(List.of(bob3), response.resources);
        }

        {
            var response = service.find(FindRequest.newBuilder()
                    .cloudId("one")
                    .selectors(Selectors.parse("resource=bob2"))
                    .build())
                    .join();

            assertEquals(List.of(bob2), response.resources);
        }
    }
}
