package ru.yandex.solomon.name.resolver.balancer;

import io.grpc.Status;
import io.grpc.Status.Code;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.balancer.BalancerProto;
import ru.yandex.solomon.balancer.TAssignShardRequest;
import ru.yandex.solomon.balancer.TUnassignShardRequest;
import ru.yandex.solomon.locks.ReadOnlyDistributedLockStub;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShardFactoryStub;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.util.host.HostUtils;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class AssignmentGateTest {
    private NameResolverShardFactoryStub stub;
    private NameResolverLocalShards shards;
    private AssignmentGate gate;
    private ReadOnlyDistributedLockStub lock;

    @Before
    public void setUp() throws Exception {
        stub = new NameResolverShardFactoryStub();
        shards = new NameResolverLocalShards();
        lock = new ReadOnlyDistributedLockStub(new ManualClock());
        gate = new AssignmentGate(shards, lock, stub);
    }

    @After
    public void tearDown() {
        stub.close();
    }

    @Test
    public void successAssign() {
        var expectSeqNo = nextSeqNo();
        gate.assignShard(TAssignShardRequest.newBuilder()
                .setAssignmentSeqNo(BalancerProto.toProto(expectSeqNo))
                .setShardId("alice")
                .build())
                .join();

        var shard = shards.getShardById("alice");
        assertNotNull(shard);
        assertEquals("alice", shard.cloudId);
        assertEquals(expectSeqNo, shard.seqNo);
    }

    @Test
    public void successUnassign() {
        var expectSeqNo = new AssignmentSeqNo(1, 42);
        var shardId = "alice";
        shards.addShard(stub.create(shardId, expectSeqNo));
        gate.unassignShard(TUnassignShardRequest.newBuilder()
                .setAssignmentSeqNo(BalancerProto.toProto(nextSeqNo()))
                .setShardId(shardId)
                .build())
                .join();

        assertNull(shards.getShardById(shardId));
    }

    @Test
    public void successUnassignAlreadyUnassign() {
        var shardId = "alice";
        gate.unassignShard(TUnassignShardRequest.newBuilder()
                .setAssignmentSeqNo(BalancerProto.toProto(nextSeqNo()))
                .setShardId(shardId)
                .build())
                .join();

        assertNull(shards.getShardById(shardId));
    }

    @Test
    public void rejectAssignFromPreviousLeader() {
        var req = TAssignShardRequest.newBuilder()
                .setAssignmentSeqNo(BalancerProto.toProto(nextSeqNo()))
                .setShardId("alice")
                .build();
        lock.setOwner("anotherHost");
        var status = gate.assignShard(req)
                .thenApply(ignore -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertEquals(status.toString(), Code.ABORTED, status.getCode());
    }

    @Test
    public void rejectUnassignFromPreviousLeader() {
        var shardId = "alice";
        shards.addShard(stub.create(shardId, nextSeqNo()));
        var req = TUnassignShardRequest.newBuilder()
                .setAssignmentSeqNo(BalancerProto.toProto(nextSeqNo()))
                .setShardId(shardId)
                .build();
        lock.setOwner("anotherHost");
        var status = gate.unassignShard(req)
                .thenApply(ignore -> Status.OK)
                .exceptionally(Status::fromThrowable)
                .join();

        assertEquals(status.toString(), Code.ABORTED, status.getCode());
    }

    private AssignmentSeqNo nextSeqNo() {
        return new AssignmentSeqNo(lock.setOwner(HostUtils.getFqdn()), System.nanoTime());
    }
}
