package ru.yandex.solomon.name.resolver.balancer;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShardFactoryStub;
import ru.yandex.solomon.util.file.SimpleFileStorage;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class AssignmentStoreTest {

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    private Path storage;
    private NameResolverLocalShards shards;
    private NameResolverShardFactoryStub factory;

    @Before
    public void setUp() throws IOException {
        storage = tmp.newFolder().toPath();
        shards = new NameResolverLocalShards();
        factory = new NameResolverShardFactoryStub();
    }

    @After
    public void tearDown() {
        factory.close();
    }

    @Test
    public void fileNotExist() throws IOException, InterruptedException {
        Files.deleteIfExists(storage);
        create().awaitAct();
        assertEquals(List.of(), shards.stream().collect(Collectors.toList()));
    }

    @Test
    public void fileEmpty() throws InterruptedException {
        var store = create();
        store.awaitAct();
        store = create();
        store.awaitAct();
        assertEquals(List.of(), shards.stream().collect(Collectors.toList()));
    }

    @Test
    public void fileCorrupted() throws InterruptedException, IOException {
        var store = create();
        shards.addShard(factory.create("alice", new AssignmentSeqNo(1, 2)));
        store.awaitAct();
        try (var stream = Files.list(storage)) {
            var it = stream.iterator();
            while (it.hasNext()) {
                var path = it.next();
                try {
                    Files.writeString(path, "corrupted");
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }

        store = create();
        store.awaitAct();
        assertEquals(List.of(), shards.stream().collect(Collectors.toList()));
    }

    @Test
    public void restoreAssignments() throws InterruptedException {
        var store = create();

        var seqNo = new AssignmentSeqNo(22, 42);
        var shardId = "alice";
        shards.addShard(factory.create(shardId, seqNo));
        store.awaitAct();
        store = create();
        store.awaitAct();

        var shard = shards.getShardById(shardId);
        assertNotNull(shard);
        assertEquals(shard.cloudId, shardId);
        assertEquals(shard.seqNo, seqNo);
    }

    @Test
    public void restoreMultipleAssignments() throws InterruptedException {
        var store = create();

        var alice = new AssignmentSeqNo(22, 42);
        shards.addShard(factory.create("alice", alice));
        store.awaitAct();

        var bob = new AssignmentSeqNo(22, 43);
        shards.addShard(factory.create("bob", bob));
        store.awaitAct();

        var eva = new AssignmentSeqNo(22, 44);
        shards.addShard(factory.create("eva", eva));
        store.awaitAct();

        shards.remove(shards.getShardById("eva"));
        store.awaitAct();

        store = create();
        store.awaitAct();

        var aliceShard = shards.getShardById("alice");
        assertNotNull(aliceShard);
        assertEquals(aliceShard.seqNo, alice);

        var bobShard = shards.getShardById("bob");
        assertNotNull(bobShard);
        assertEquals(bobShard.seqNo, bob);

        assertNull(shards.getShardById("eva"));
    }

    private AssignmentStore create() {
        shards = new NameResolverLocalShards();
        return new AssignmentStore(new SimpleFileStorage(storage), shards, ForkJoinPool.commonPool(), factory);
    }


}
