package ru.yandex.solomon.name.resolver.balancer;

import java.util.List;
import java.util.Set;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.locks.ReadOnlyDistributedLockStub;
import ru.yandex.solomon.name.resolver.db.InMemoryShardsDao;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverShardsHolderTest {
    private ManualClock clock;
    private ReadOnlyDistributedLockStub lock;
    private InMemoryShardsDao shardsDao;
    private ShardsHolder holder;

    @Before
    public void setUp() {
        clock = new ManualClock();
        lock = new ReadOnlyDistributedLockStub(clock);
        lock.setOwner(HostUtils.getFqdn());
        shardsDao = new InMemoryShardsDao();
        holder = new NameResolverShardsHolder(shardsDao, lock);
    }

    @Test
    public void emptyShards() {
        holder.reload().join();
        assertEquals(Set.of(), holder.getShards());
    }

    @Test
    public void reloadShards() {
        holder.reload().join();
        shardsDao.insert("alice").join();
        holder.reload().join();
        assertEquals(Set.of("alice"), holder.getShards());

        shardsDao.insert("bob").join();
        holder.reload().join();
        assertEquals(Set.of("alice", "bob"), holder.getShards());
    }

    @Test
    public void addShard() {
        holder.reload().join();
        add("alice");
        assertEquals(Set.of("alice"), holder.getShards());
        assertEquals(List.of("alice"), shardsDao.findAll().join());
    }

    @Test
    public void deleteShard() {
        holder.reload().join();
        add("alice");
        holder.delete("alice").join();
        assertEquals(Set.of(), holder.getShards());
        assertEquals(List.of(), shardsDao.findAll().join());
    }

    @Test
    public void deleteNotExist() {
        holder.reload().join();
        add("alice");
        holder.delete("bob").join();
        assertEquals(Set.of("alice"), holder.getShards());
        assertEquals(List.of("alice"), shardsDao.findAll().join());
    }

    @Test
    public void addDeleteAdd() {
        holder.reload().join();
        add("alice");
        holder.delete("alice"); // don't wait complete
        add("alice");
        assertEquals(Set.of("alice"), holder.getShards());
        assertEquals(List.of("alice"), shardsDao.findAll().join());
    }

    @Test
    public void deleteMultipleTimes() {
        holder.reload().join();
        add("alice");
        IntStream.range(1, 100)
                .parallel()
                .mapToObj(ignore -> holder.delete("alice"))
                .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
                .join();
        assertEquals(Set.of(), holder.getShards());
        assertEquals(List.of(), shardsDao.findAll().join());
    }

    @Test
    public void unableDeleteWhenNotLeader() {
        holder.reload().join();
        add("alice");
        lock.setOwner(null);

        var exception = holder.delete("alice")
                .thenApply(unused -> null)
                .exceptionally(e -> e)
                .join();

        assertNotNull(exception);
        assertEquals(List.of("alice"), shardsDao.findAll().join());
    }

    @Test
    public void removeFromDeleteListWhenDeleteInFlight() {
        holder.reload().join();
        add("alice");
        assertEquals(Set.of("alice"), holder.getShards());
        holder.delete("alice"); // don't wait complete
        assertEquals(Set.of(), Set.copyOf(holder.getShards()));
    }

    private void add(String shardId) {
        holder.add(shardId).join();
    }
}
