package ru.yandex.solomon.name.resolver.db;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

import ru.yandex.solomon.name.resolver.client.Resource;

/**
 * @author Vladimir Gordiychuk
 */
public class InMemoryResourceDao implements ResourcesDao {
    private final ConcurrentMap<String, CloudStore> storeByCloudId = new ConcurrentHashMap<>();
    private volatile Throwable throwable;

    public void setError(Throwable e) {
        throwable = e;
    }

    @Override
    public CompletableFuture<Void> replaceResources(Collection<Resource> resources) {
        return CompletableFuture.runAsync(() -> {
            ensureNoError();
            for (var resource : resources) {
                storeByCloudId.computeIfAbsent(resource.cloudId, CloudStore::new).updateResource(resource);
            }
        });
    }

    @Override
    public CompletableFuture<Void> deleteResources(Collection<Resource> resources) {
        return CompletableFuture.runAsync(() -> {
            ensureNoError();
            for (var resource : resources) {
                storeByCloudId.computeIfAbsent(resource.cloudId, CloudStore::new).deleteResource(resource);
            }
        });
    }

    @Override
    public CompletableFuture<List<Resource>> findResources(String cloudId) {
        return CompletableFuture.supplyAsync(() -> {
            ensureNoError();
            return storeByCloudId.computeIfAbsent(cloudId, CloudStore::new).list();
        });
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return CompletableFuture.runAsync(this::ensureNoError);
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return CompletableFuture.runAsync(storeByCloudId::clear);
    }

    private void ensureNoError() {
        var copy = throwable;
        if (copy != null) {
            throw new RuntimeException(copy);
        }
    }

    private static class CloudStore {
        private final String cloudId;
        private final ConcurrentMap<String, Resource> resourceById = new ConcurrentHashMap<>();

        public CloudStore(String cloudId) {
            this.cloudId = cloudId;
        }

        public void updateResource(Resource resource) {
            resourceById.put(resource.resourceId, new Resource(resource));
        }

        public void deleteResource(Resource resource) {
            resourceById.remove(resource.resourceId);
        }

        public List<Resource> list() {
            return resourceById.values().stream().map(Resource::new).collect(Collectors.toList());
        }
    }
}
