package ru.yandex.solomon.name.resolver.db;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Test;

import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class ResourcesDaoTest {
    protected abstract ResourcesDao getDao();

    @Test
    public void createSchemaMultipleTimes() {
        getDao().createSchemaForTests().join();
        getDao().createSchemaForTests().join();
        getDao().createSchemaForTests().join();
    }

    @Test
    public void findEmpty() {
        assertEquals(List.of(), getDao().findResources("notExistsCloudId").join());
    }

    @Test
    public void updateResource() {
        var dao = getDao();
        var v1 = staticResource();

        dao.replaceResources(List.of(v1)).join();
        assertEquals(List.of(v1), dao.findResources(v1.cloudId).join());

        var v2 = new Resource(v1)
                .setName("UpdatedResourceName")
                .setUpdatedAt(System.currentTimeMillis());

        dao.replaceResources(List.of(v2)).join();
        assertEquals(List.of(v2), dao.findResources(v2.cloudId).join());

        var v3 = new Resource(v2);
        assertEquals(List.of(v3), dao.findResources(v3.cloudId).join());
    }

    @Test
    public void replaceBySame() {
        var resource = new Resource()
                .setCloudId("aoe9shbqc2v314v7fp3d")
                .setFolderId("aoed5i52uquf5jio0oec")
                .setService("managed-clickhouse")
                .setType("cluster")
                .setResourceId("e4u11un3hq311tafkquh")
                .setName("dbaas_e2e_compute_preprod")
                .setResourceComplexId(Map.of("a", "b"))
                .setSeverity(Resource.Severity.CRITICAL)
                .setResponsible(UUID.randomUUID().toString())
                .setEnvironment(UUID.randomUUID().toString())
                .setUpdatedAt(1606487008399L)
                .setDeletedAt(0);

        var dao = getDao();
        dao.replaceResources(List.of(resource)).join();
        dao.replaceResources(List.of(resource)).join();
        assertEquals(List.of(resource), dao.findResources(resource.cloudId).join());
    }

    @Test
    public void filterByCloudId() {
        var dao = getDao();
        var alice = staticResource().setCloudId("alice").setResourceId("1");
        var bob = staticResource().setCloudId("bob").setResourceId("2");

        dao.replaceResources(List.of(alice)).join();
        dao.replaceResources(List.of(bob)).join();

        assertEquals(List.of(alice), dao.findResources("alice").join());
        assertEquals(List.of(bob), dao.findResources("bob").join());
    }

    @Test
    public void deleteResources() {
        var dao = getDao();
        var alice = staticResource().setResourceId("alice");
        var bob = staticResource().setResourceId("bob");

        dao.replaceResources(List.of(alice, bob)).join();
        assertEquals(Set.of(alice, bob), Set.copyOf(dao.findResources(alice.cloudId).join()));

        dao.deleteResources(List.of(alice)).join();
        assertEquals(List.of(bob), dao.findResources(alice.cloudId).join());
    }

    @Test
    public void deleteMany() {
        var dao = getDao();
        var expected = IntStream.range(0, 10000)
                .mapToObj(id -> staticResource().setResourceId(Integer.toString(id)))
                .collect(Collectors.toList());

        Collections.shuffle(expected);
        for (var batch : Lists.partition(expected, 1000)) {
            dao.replaceResources(batch).join();
        }

        Collections.shuffle(expected);
        for (var batch : Lists.partition(expected, 1000)) {
            dao.deleteResources(batch).join();
        }

        var result = dao.findResources(expected.get(0).cloudId).join();
        assertEquals(List.of(), result);
    }

    @Test
    public void updateManyFindMany() {
        var dao = getDao();
        var expected = IntStream.range(0, 10000)
                .mapToObj(id -> staticResource().setResourceId(Integer.toString(id)))
                .collect(Collectors.toList());

        Collections.shuffle(expected);
        for (var batch : Lists.partition(expected, 1000)) {
            dao.replaceResources(batch).join();
        }

        // noise
        dao.replaceResources(IntStream.range(0, 100)
                .mapToObj(id -> staticResource()
                        .setResourceId("noise-" + id)
                        .setCloudId(RandomStringUtils.randomAlphanumeric(10)))
                .collect(Collectors.toList()))
                .join();

        var result = dao.findResources(expected.get(0).cloudId).join();
        expected.sort(Comparator.comparing(o -> o.resourceId));
        result.sort(Comparator.comparing(o -> o.resourceId));
        assertArrayEquals(expected.toArray(), result.toArray());
    }

    @Test
    public void saveReindexAt() {
        var dao = getDao();
        var v1 = staticResource().setReindexAt(System.currentTimeMillis());

        dao.replaceResources(List.of(v1)).join();
        assertEquals(List.of(v1), dao.findResources(v1.cloudId).join());
    }
}
