package ru.yandex.solomon.name.resolver.index;

import java.util.HashSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.stream.Collectors;

import org.junit.Test;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;


/**
 * @author Sergey Polovko
 * @author Vladimir Gordiychuk
 */
public class LevelTest {

    @Test
    public void updateEmptyLevel() {
        var resource = staticResource().setName("alice");

        Level level = new Level();
        Level updatedLevel = level.update(List.of(resource));

        assertNotSame(level, updatedLevel);

        // initial level not changed
        assertEquals(0, level.size());

        assertEquals(1, updatedLevel.size());
        assertEquals(List.of(resource), search(updatedLevel, "resource=alice"));
    }

    @Test
    public void emptyIterator() {
        var level = new Level();
        var it = level.iterator();
        assertFalse(it.hasNext());
        try {
            it.next();
            fail();
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void nonEmptyIterator() {
        var resources = List.of(
                staticResource().setResourceId("a").setName("alice"),
                staticResource().setResourceId("b").setName("bob"),
                staticResource().setResourceId("c").setName("eva"));

        var level = new Level(resources);
        var it = level.iterator();
        Set<Resource> expected = new HashSet<>(resources);
        while (it.hasNext()) {
            var resource = it.next();
            assertTrue(resource.toString(), expected.remove(resource));
        }
        assertTrue(expected.toString(), expected.isEmpty());
        try {
            it.next();
            fail();
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void updateFilledLevel() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        Level level = new Level(List.of(alice));
        Level updatedLevel = level.update(List.of(bob));

        assertNotSame(level, updatedLevel);

        // initial level not changed
        assertEquals(1, level.size());

        assertEquals(2, updatedLevel.size());
        assertEquals(List.of(alice), search(updatedLevel, "resource=alice"));
        assertEquals(List.of(bob), search(updatedLevel, "resource=bob"));
    }

    @Test
    public void updateEmpty() {
        var resource = staticResource();
        Level level = new Level(List.of(resource));
        Level updatedLevel = level.update(List.of());

        // initial level not changed
        assertEquals(1, level.size());

        assertSame(level, updatedLevel);
    }

    @Test
    public void updateChangeName() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("one").setName("bob");

        Level level = new Level(List.of(alice));
        Level updatedLevel = level.update(List.of(bob));

        assertNotSame(level, updatedLevel);

        // initial level not changed
        assertEquals(1, level.size());

        assertEquals(1, updatedLevel.size());
        assertEquals(List.of(), search(updatedLevel, "resource=alice"));
        assertEquals(List.of(bob), search(updatedLevel, "resource=bob"));
    }

    @Test
    public void mergeLeftEmpty() {
        var resource = staticResource().setName("alice");
        Level left = new Level();
        Level right = new Level(List.of(resource));
        Level merged = left.mergeWith(right, false);

        // left not changed
        assertEquals(0, left.size());
        // right not changed
        assertEquals(1, right.size());

        assertSame(merged, right);
    }

    @Test
    public void mergeRightEmpty() {
        var resource = staticResource().setName("alice");

        Level left = new Level(List.of(resource));
        Level right = new Level();
        Level merged = left.mergeWith(right, false);

        // left not changed
        assertEquals(1, left.size());
        // right not changed
        assertEquals(0, right.size());

        assertSame(merged, left);
    }

    @Test
    public void merge() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        Level left = new Level(List.of(alice));
        Level right = new Level(List.of(bob));
        Level merged = left.mergeWith(right, false);

        // left not changed
        assertEquals(1, left.size());
        // right not changed
        assertEquals(1, right.size());

        assertNotSame(merged, left);
        assertNotSame(merged, right);
        assertEquals(2, merged.size());

        assertEquals(List.of(alice), search(merged, "resource=alice"));
        assertEquals(List.of(bob), search(merged, "resource=bob"));
    }

    @Test
    public void mergeChangeName() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("one").setName("bob");

        Level left = new Level(List.of(alice));
        Level right = new Level(List.of(bob));
        Level merged = left.mergeWith(right, false);

        // left not changed
        assertEquals(1, left.size());
        // right not changed
        assertEquals(1, right.size());

        assertNotSame(merged, left);
        assertNotSame(merged, right);
        assertEquals(1, merged.size());

        assertEquals(List.of(), search(merged, "resource=alice"));
        assertEquals(List.of(bob), search(merged, "resource=bob"));
    }

    @Test
    public void searchDistinct() {
        var resource = staticResource().setName("alice");

        Level level = new Level();
        Level updatedLevel = level.update(List.of(resource));

        assertNotSame(level, updatedLevel);

        // initial level not changed
        assertEquals(0, level.size());

        assertEquals(1, updatedLevel.size());
        assertEquals(List.of(resource), search(updatedLevel, "resource=*"));
    }

    @Test
    public void hasRemoved() {
        var resource = staticResource().setName("alice");

        var level = new Level(List.of(resource));
        assertTrue(level.has(resource.resourceId));

        var updated = remove(level, resource.resourceId);
        assertTrue(updated.has(resource.resourceId));
    }

    @Test
    public void getRemoved() {
        var resource = staticResource().setName("alice");

        var level = new Level(List.of(resource));
        assertEquals(resource, level.getResourceById(resource.resourceId));

        var updated = remove(level, resource.resourceId);
        assertNull(updated.getResourceById(resource.resourceId));
    }

    @Test
    public void searchRemoved() {
        var resource = staticResource().setName("alice");

        var level = new Level(List.of(resource));
        assertEquals(List.of(resource), search(level, "resource=alice"));

        var updated = remove(level, resource.resourceId);
        assertEquals(List.of(), search(updated, "resource=alice"));
    }

    @Test
    public void iteratorRemoved() {
        var resource = staticResource().setName("alice");

        var level = new Level(List.of(resource));
        {
            var it = level.iterator();
            assertTrue(it.hasNext());
            assertEquals(resource, it.next());
            assertFalse(it.hasNext());
        }

        var updated = remove(level, resource.resourceId);
        {
            var it = updated.iterator();
            assertFalse(it.hasNext());
        }
    }

    @Test
    public void mergeRemoved() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("one").setName("bob");
        var eva = staticResource().setResourceId("two").setName("eva");

        Level left = remove(new Level(List.of(alice)), alice.resourceId);
        Level right = new Level(List.of(bob, eva));
        Level merged = left.mergeWith(right, true);

        // left not changed
        assertEquals(1, left.size());
        // right not changed
        assertEquals(2, right.size());

        assertNotSame(merged, left);
        assertNotSame(merged, right);
        // alice\bob removed
        assertEquals(1, merged.size());

        assertEquals(List.of(), search(merged, "resource=alice"));
        assertEquals(List.of(), search(merged, "resource=bob"));
        assertEquals(List.of(eva), search(merged, "resource=eva"));
    }

    private Level remove(Level level, String... resourceIds) {
        return level.remove(List.of(new RemoveRequest(List.of(resourceIds))));
    }

    private List<Resource> search(Level level, String selector) {
        return level.search(Selectors.parse(selector))
                .collect(Collectors.toList());
    }
}
