package ru.yandex.solomon.name.resolver.index;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import org.junit.Test;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;


/**
 * @author Sergey Polovko
 * @author Vladimir Gordiychuk
 */
public class LevelsArrayTest {

    private static final int[] TEST_LEVELS_SIZES = { 1, 2, 4, 8 };

    @Test
    public void emptyArray() {
        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        assertLevelsSize(a, 0, 0, 0, 0);
        assertEquals(0, a.size());
    }

    @Test
    public void updateEmptyDiff() {
        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        LevelsArray b = a.updateZeroLevel(List.of());
        assertSame(a, b);
    }

    @Test
    public void updateEmptyLevels() {
        var resource = staticResource().setName("alice");

        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        LevelsArray b = a.updateZeroLevel(List.of(resource));
        assertNotSame(a, b);

        // a not changed
        assertLevelsSize(a, 0, 0, 0, 0);
        assertEquals(0, a.size());

        // b has new resources
        assertLevelsSize(b, 1, 0, 0, 0);
        assertEquals(1, b.size());
        assertEquals(List.of(resource), search(b, "resource=alice"));
        assertEquals(List.of(), search(b, "resource=bob"));
    }

    @Test
    public void updateNonEmptyLevels() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, List.of(alice));
        LevelsArray b = a.updateZeroLevel(List.of(bob));
        assertNotSame(a, b);

        // a not changed
        assertLevelsSize(a, 0, 0, 0, 1);
        assertEquals(1, a.size());

        // b has new resources
        assertLevelsSize(b, 1, 0, 0, 1);
        assertEquals(2, b.size());

        assertEquals(List.of(alice), search(b, "resource=alice"));
        assertEquals(List.of(bob), search(b, "resource=bob"));
    }

    @Test
    public void skipMergeOverfullLevel() {
        var resource = staticResource();

        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        LevelsArray b = a.updateZeroLevel(List.of(resource));
        LevelsArray c = b.mergeOverfullLevel();

        // b not changed
        assertLevelsSize(b, 1, 0, 0, 0);
        assertEquals(1, b.size());

        assertSame(b, c);
    }

    @Test
    public void mergeFromZeroToNextLevel() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        LevelsArray a = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        LevelsArray b = a.updateZeroLevel(List.of(alice, bob));
        LevelsArray c = b.mergeOverfullLevel();

        // b not changed
        assertLevelsSize(b, 2, 0, 0, 0);
        assertEquals(2, b.size());

        assertNotSame(b, c);
        assertLevelsSize(c, 0, 2, 0, 0);
        assertEquals(2, c.size());

        assertEquals(List.of(alice), search(c, "resource=alice"));
        assertEquals(List.of(bob), search(c, "resource=bob"));
    }

    @Test
    public void mergeOverfullLevel() {
        LevelsArray array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        List<Resource> expectedResource = new ArrayList<>();
        for (int index = 0; index < 100; index++) {
            var resource = staticResource()
                    .setFolderId("folder-"+index)
                    .setService("service-"+index)
                    .setType("type-"+index)
                    .setResourceId("resourceId-" + index)
                    .setName("name-"+index);
            expectedResource.add(new Resource(resource));
            array = array.updateZeroLevel(List.of(resource));
            array = array.mergeOverfullLevel();
        }

        var actual = search(array, "resource=*");
        actual.sort(Comparator.comparing(o -> o.resourceId));
        expectedResource.sort(Comparator.comparing(o -> o.resourceId));

        assertArrayEquals(expectedResource.toArray(), actual.toArray());
    }

    @Test
    public void updateResourceName() {
        LevelsArray array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("one").setName("bob");

        array = array.updateZeroLevel(List.of(alice));
        // noise to move alice to to latest level
        for (int index = 0; index < 100; index++) {
            var resource = staticResource()
                    .setFolderId("folder-"+index)
                    .setService("service-"+index)
                    .setType("type-"+index)
                    .setResourceId("resourceId-" + index)
                    .setName("name-"+index);
            array = array.updateZeroLevel(List.of(resource));
            array = array.mergeOverfullLevel();
        }

        // alice name changed to bob
        array = array.updateZeroLevel(List.of(bob));
        assertEquals(List.of(bob), search(array, "resource=bob"));
    }

    @Test
    public void matchOnlyByLatestValue() {
        LevelsArray array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        Resource latest = null;
        for (int index = 0; index < 100; index++) {
            latest = staticResource().setResourceId("myId").setName("name-" + index);
            array = array.updateZeroLevel(List.of(latest));
            array = array.mergeOverfullLevel();
        }

        assertEquals(List.of(latest), search(array,"resource=name-99"));
        assertEquals(List.of(latest), search(array,"resource=*"));
        assertEquals(List.of(), search(array,"resource=name-42"));
    }

    @Test
    public void emptyIterator() {
        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        var it = array.iterator();
        assertFalse(it.hasNext());
        try {
            it.next();
            fail();
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void oneLevelIterator() {
        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        var expected = staticResource();
        array = array.updateZeroLevel(List.of(expected));
        var it = array.iterator();
        assertTrue(it.hasNext());
        assertEquals(expected, it.next());
        assertFalse(it.hasNext());
        try {
            it.next();
            fail();
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void crossLevelIterator() {
        LevelsArray array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        List<Resource> expectedResource = new ArrayList<>();
        for (int index = 0; index < 100; index++) {
            var resource = staticResource()
                    .setFolderId("folder-"+index)
                    .setService("service-"+index)
                    .setType("type-"+index)
                    .setResourceId("resourceId-" + index)
                    .setName("name-"+index);
            expectedResource.add(new Resource(resource));
            array = array.updateZeroLevel(List.of(resource));
            array = array.mergeOverfullLevel();
        }

        Set<Resource> notVisit = new HashSet<>(expectedResource);
        for (Resource resource : array) {
            assertTrue(resource.toString(), notVisit.remove(resource));
        }

        assertTrue(notVisit.isEmpty());
    }

    @Test
    public void iteratorReturnLatestResourceUpdate() {
        LevelsArray array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        Resource latest = null;
        for (int index = 0; index < 100; index++) {
            latest = staticResource().setResourceId("myId").setName("name-" + index);
            array = array.updateZeroLevel(List.of(latest));
            array = array.mergeOverfullLevel();
        }

        var it = array.iterator();
        assertTrue(it.hasNext());
        assertEquals(latest, it.next());
        assertFalse(it.hasNext());
    }

    @Test
    public void hasRemoved() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of(alice, bob));
        assertTrue(array.has(alice.resourceId));

        var updated = remove(array, alice.resourceId);
        assertTrue(updated.has(alice.resourceId));
    }

    @Test
    public void getRemoved() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of(alice, bob));
        assertEquals(alice, array.getOrNull(alice.resourceId));

        var updated = remove(array, alice.resourceId);
        assertNull(updated.getOrNull(alice.resourceId));
    }

    @Test
    public void searchRemoved() {
        var alice = staticResource().setResourceId("one").setName("alice");
        var bob = staticResource().setResourceId("two").setName("bob");

        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of(alice, bob));
        assertEquals(List.of(alice), search(array, "resource=alice"));

        var updated = remove(array, alice.resourceId);
        assertEquals(List.of(), search(updated, "resource=alice"));
        assertEquals(List.of(bob), search(updated, ""));
    }

    @Test
    public void addRemove() {
        var alice = staticResource().setResourceId("one").setName("alice");

        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        array = update(array, alice);
        assertEquals(List.of(alice), search(array, "resource=alice"));
        assertEquals(List.of(alice), search(array, ""));

        array = remove(array, alice.resourceId);
        assertEquals(List.of(), search(array, "resource=alice"));
        assertEquals(List.of(), search(array, ""));
    }

    @Test
    public void iteratorRemoved() {
        var alice = staticResource().setResourceId("one").setName("alice");

        var array = new LevelsArray(TEST_LEVELS_SIZES, List.of(alice));
        {
            var it = array.iterator();
            assertTrue(it.hasNext());
            assertEquals(alice, it.next());
            assertFalse(it.hasNext());
        }

        var updated = remove(array, alice.resourceId);
        {
            var it = updated.iterator();
            assertFalse(it.hasNext());
        }
    }

    @Test
    public void mergeRemoved() {
        LevelsArray array = new LevelsArray(TEST_LEVELS_SIZES, List.of());
        List<Resource> resources = new ArrayList<>();
        for (int index = 0; index < 100; index++) {
            var resource = staticResource()
                    .setFolderId("folder-"+index)
                    .setService("service-"+index)
                    .setType("type-"+index)
                    .setResourceId("resourceId-" + index)
                    .setName("name-"+index);
            resources.add(new Resource(resource));
            array = update(array, resource);
            array = array.mergeOverfullLevel();
            if (index >= 5) {
                int removeIdx = index - 5;
                array = remove(array, resources.get(removeIdx).resourceId);
                var removed = resources.get(ThreadLocalRandom.current().nextInt(removeIdx + 1));
                assertEquals(List.of(), search(array, "resource="+removed.name));
            }
        }

        Set<Resource> notVisit = new HashSet<>(resources.subList(95, 100));
        for (Resource resource : array) {
            assertTrue(resource.toString(), notVisit.remove(resource));
        }

        assertTrue(notVisit.toString(), notVisit.isEmpty());
    }

    private static void assertLevelsSize(LevelsArray a, int... sizes) {
        assertEquals(a.count(), sizes.length);
        for (int i = 0; i < a.count(); i++) {
            assertEquals("at pos " + i, sizes[i], a.levelSize(i));
        }
    }

    private LevelsArray remove(LevelsArray array, String... resourceIds) {
        return array.remove(List.of(new RemoveRequest(List.of(resourceIds))));
    }

    private LevelsArray update(LevelsArray array, Resource... resources) {
        return array.updateZeroLevel(Arrays.asList(resources));
    }

    private List<Resource> search(LevelsArray array, String selector) {
        return array.search(Selectors.parse(selector))
                .collect(Collectors.toList());
    }
}
