package ru.yandex.solomon.name.resolver.index;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;

import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.ResourceLabels;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceLabelsIteratorTest {

    @Test
    public void empty() {
        var it = new ResourceLabelsIterator(Collections.emptyIterator());
        assertFalse(it.hasNext());
        try {
            it.next();
            fail();
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void oneResourceOneLabels() {
        var resource = staticResource();
        var it = new ResourceLabelsIterator(List.of(resource).iterator());
        assertTrue(it.hasNext());
        assertEquals(Labels.builder()
                .add(ResourceLabels.FOLDER_ID, resource.folderId)
                .add(ResourceLabels.SERVICE, resource.service)
                .add(ResourceLabels.TYPE, resource.type)
                .add(ResourceLabels.RESOURCE, resource.name)
                .add(ResourceLabels.NAME, resource.name)
                .add(ResourceLabels.ID, resource.resourceId)
                .add(ResourceLabels.SEVERITY, resource.severity.name())
                .build(), it.next());

        assertFalse(it.hasNext());
    }

    @Test
    public void manyResources() {
        List<Labels> expected = new ArrayList<>();
        List<Resource> resources = new ArrayList<>();
        for (int index = 0; index < 10; index++) {
            var resource = staticResource()
                    .setFolderId("folder-" + index)
                    .setService("service-" + index)
                    .setType("type-" + index)
                    .setResourceId("resourceId-" + index)
                    .setName("name-" + index)
                    .setSeverity(Resource.Severity.NON_CRITICAL);

            expected.add(Labels.builder()
                    .add(ResourceLabels.FOLDER_ID, resource.folderId)
                    .add(ResourceLabels.SERVICE, resource.service)
                    .add(ResourceLabels.TYPE, resource.type)
                    .add(ResourceLabels.RESOURCE, resource.name)
                    .add(ResourceLabels.ID, resource.resourceId)
                    .add(ResourceLabels.NAME, resource.name)
                    .add(ResourceLabels.SEVERITY, resource.severity.name())
                    .build());
            resources.add(resource);
        }

        var it = new ResourceLabelsIterator(resources.iterator());
        List<Labels> actual = new ArrayList<>();
        while (it.hasNext()) {
            actual.add(it.next());
        }
        assertArrayEquals(expected.toArray(), actual.toArray());
    }
}
