package ru.yandex.solomon.name.resolver.index;

import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.junit.Test;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;


/**
 * @author Sergey Polovko
 */
public class ResourcesCollectionImpTest {
    private static final int[] LEVELS_SIZES = { 1, 2, 10 };
    private final Executor executor = ForkJoinPool.commonPool();

    @Test
    public void getOrNull() {
        var collection = new ResourcesCollectionImpl(
            "test", executor, LEVELS_SIZES, List.of(
                    resource("a"),
                    resource("b")));

        assertEquals(2, collection.size());

        assertNotNull(collection.getOrNull("a"));
        assertNotNull(collection.getOrNull("b"));

        assertNull(collection.getOrNull("c"));
        assertNull(collection.getOrNull("d"));
    }

    @Test
    public void has() {
        var collection = new ResourcesCollectionImpl(
                "test", executor, LEVELS_SIZES, List.of(
                resource("a"),
                resource("b")));

        assertEquals(2, collection.size());

        assertTrue(collection.has("a"));
        assertTrue(collection.has("b"));

        assertFalse(collection.has("c"));
        assertFalse(collection.has("d"));
    }

    @Test
    public void putAll() {
        var collection = new AwaitableResourcesCollection(
                executor, LEVELS_SIZES, List.of(
                resource("a"),
                resource("b")));
        assertEquals(2, collection.size());
        assertEquals(0, collection.levelSize(0));
        assertEquals(0, collection.levelSize(1));
        assertEquals(2, collection.levelSize(2));

        collection.putAll(List.of(resource("c"), resource("d")));
        collection.awaitActDone();

        assertEquals(4, collection.size());
        assertEquals(0, collection.levelSize(0));
        assertEquals(2, collection.levelSize(1));
        assertEquals(2, collection.levelSize(2));

        assertNotNull(collection.getOrNull("c"));
        assertNotNull(collection.getOrNull("d"));

        collection.putAll(List.of(
            resource("e"),
            resource("d"),
            resource("f")));
        collection.awaitActDone();

        assertEquals(6, collection.size());
        assertEquals(0, collection.levelSize(0));
        assertEquals(4, collection.levelSize(1));
        assertEquals(2, collection.levelSize(2));

        assertNotNull(collection.getOrNull("f"));

        collection.putAll(List.of(resource("g")));
        collection.awaitActDone();

        assertEquals(7, collection.size());
        assertEquals(1, collection.levelSize(0));
        assertEquals(0, collection.levelSize(1));
        assertEquals(6, collection.levelSize(2));

        assertNotNull(collection.getOrNull("g"));

        collection.putAll(List.of(resource("h")));
        collection.awaitActDone();

        assertEquals(8, collection.size());
        assertEquals(0, collection.levelSize(0));
        assertEquals(2, collection.levelSize(1));
        assertEquals(6, collection.levelSize(2));

        assertNotNull(collection.getOrNull("h"));
    }

    @Test
    public void searchWithCount() {
        // fill all levels
        AwaitableResourcesCollection collection = new AwaitableResourcesCollection(
            executor, LEVELS_SIZES,
            List.of(
                resource("a"),
                resource("b"),
                resource("c"),
                resource("d"),
                resource("e"),
                resource("f")));

        collection.putAll(List.of(resource("g"), resource("h")));
        collection.awaitActDone();

        collection.put(resource("i"));
        collection.awaitActDone();

        assertEquals(9, collection.size());
        assertEquals(1, collection.levelSize(0));
        assertEquals(2, collection.levelSize(1));
        assertEquals(6, collection.levelSize(2));

        // search one resource from each level
        List<Resource> result = collection.search(Selectors.parse("resource=i|a|h")).collect(Collectors.toList());
        assertEquals(3, result.size());

        Map<String, Resource> resourceById = result.stream()
            .collect(Collectors.toMap(resource -> resource.resourceId, Function.identity()));
        assertEquals(3, resourceById.size());
        assertNotNull(resourceById.get("i"));
        assertNotNull(resourceById.get("a"));
        assertNotNull(resourceById.get("h"));
    }

    @Test
    public void emptyIterator() {
        ResourcesCollection collection = new ResourcesCollectionImpl(
                "test", executor, LEVELS_SIZES, List.of());

        var it = collection.iterator();
        assertFalse(it.hasNext());
    }

    @Test
    public void initialFilledIterator() {
        List<Resource> expected = List.of(
                resource("1"),
                resource("2"),
                resource("3"));
        ResourcesCollection collection = new ResourcesCollectionImpl(
                "test", executor, LEVELS_SIZES, expected);

        Set<Resource> actual = new HashSet<>();
        collection.forEach(actual::add);
        assertEquals(Set.copyOf(expected), actual);
    }

    @Test
    public void iteratorFilledAtRuntime() {
        AwaitableResourcesCollection collection = new AwaitableResourcesCollection(
                executor, LEVELS_SIZES, List.of());

        Set<Resource> expected = new HashSet<>();

        for (int i = 0; i < 100; i++) {
            var resource = resource("a"+i);
            assertTrue(expected.add(resource));
            collection.put(resource);
            collection.awaitActDone();

            Set<Resource> actual = new HashSet<>();
            collection.forEach(actual::add);

            assertEquals(expected, actual);
        }
    }

    @Test
    public void searchRemoved() {
        AwaitableResourcesCollection collection = new AwaitableResourcesCollection(
                executor, LEVELS_SIZES,
                List.of(
                        resource("a"),
                        resource("b"),
                        resource("c"),
                        resource("d"),
                        resource("e"),
                        resource("f")));

        collection.putAll(List.of(resource("g"), resource("h")));
        collection.awaitActDone();

        collection.put(resource("i"));
        collection.awaitActDone();

        assertEquals(9, collection.size());
        collection.removeAll(List.of("i", "a", "d", "f"));
        collection.awaitActDone();

        // search one resource from each level
        List<Resource> result = collection.search(Selectors.parse("resource=i|a|h")).collect(Collectors.toList());
        // a, i was removed
        assertEquals(List.of(resource("h")), result);
    }

    @Test
    public void replaceOneResourceByOther() {
        var now = System.currentTimeMillis();
        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of());

        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        collection.put(init);
        collection.awaitActDone();
        assertEquals(init, collection.getOrNull("one"));

        var deleted = new Resource(init).setDeletedAt(now + 1).setDeletedAt(now + 1);
        collection.put(deleted);
        collection.awaitActDone();
        assertEquals(deleted, collection.getOrNull("one"));

        var created = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);
        collection.put(created);
        collection.awaitActDone();

        assertEquals(created, collection.getOrNull("two"));
        assertEquals(new Resource(deleted).setReplaced(true), collection.getOrNull("one"));
    }

    @Test
    public void replaceOneResourceByOtherDeleteLate() {
        var now = System.currentTimeMillis();
        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of());

        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        collection.put(init);
        collection.awaitActDone();
        assertEquals(init, collection.getOrNull("one"));

        var created = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);
        collection.put(created);
        collection.awaitActDone();
        assertEquals(created, collection.getOrNull("two"));
        assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));

        var deleted = new Resource(init).setDeletedAt(now + 2).setUpdatedAt(now + 1);
        collection.put(deleted);
        collection.awaitActDone();

        assertEquals(created, collection.getOrNull("two"));
        assertEquals(new Resource(deleted).setReplaced(true), collection.getOrNull("one"));
    }

    @Test
    public void replaceOneResourceByOtherAndRestoreByRenameActual() {
        var now = System.currentTimeMillis();
        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of());

        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        collection.put(init);
        collection.awaitActDone();
        assertEquals(init, collection.getOrNull("one"));

        var replace = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);
        collection.put(replace);
        collection.awaitActDone();
        assertEquals(replace, collection.getOrNull("two"));
        assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));

        var unique = new Resource(replace).setName("unique").setUpdatedAt(now + 3);
        collection.put(unique);
        collection.awaitActDone();
        assertEquals(unique, collection.getOrNull("two"));
        assertEquals(init, collection.getOrNull("one"));
    }

    @Test
    public void replaceOneResourceByOtherRestoreByRenameReplaced() {
        var now = System.currentTimeMillis();
        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of());

        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        collection.put(init);
        collection.awaitActDone();
        assertEquals(init, collection.getOrNull("one"));

        var replace = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);
        collection.put(replace);
        collection.awaitActDone();
        assertEquals(replace, collection.getOrNull("two"));
        assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));

        var unique = new Resource(init).setName("unique").setUpdatedAt(now + 3);
        collection.put(unique);
        collection.awaitActDone();
        assertEquals(replace, collection.getOrNull("two"));
        assertEquals(unique, collection.getOrNull("one"));
    }

    @Test
    public void replaceOneResourceByOtherRestoreOnlyOne() {
        var now = System.currentTimeMillis();
        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of());

        var replacedOne = resource("one").setName("not-unique-name").setUpdatedAt(now);
        collection.put(replacedOne);
        collection.awaitActDone();
        assertEquals(replacedOne, collection.getOrNull("one"));

        var replacedTwo = resource("two").setName("not-unique-name").setUpdatedAt(now + 1);
        collection.put(replacedTwo);
        collection.awaitActDone();
        assertEquals(replacedTwo, collection.getOrNull("two"));
        assertEquals(new Resource(replacedOne).setReplaced(true), collection.getOrNull("one"));

        var tree = resource("tree").setName("not-unique-name").setUpdatedAt(now + 2);
        collection.put(tree);
        collection.awaitActDone();
        assertEquals(tree, collection.getOrNull("tree"));
        assertEquals(new Resource(replacedOne).setReplaced(true), collection.getOrNull("one"));
        assertEquals(new Resource(replacedTwo).setReplaced(true), collection.getOrNull("two"));

        var treeUnique = new Resource(tree).setName("unique").setUpdatedAt(now + 3);
        collection.put(treeUnique);
        collection.awaitActDone();

        assertEquals(treeUnique, collection.getOrNull("tree"));
        assertEquals(new Resource(replacedOne).setReplaced(true), collection.getOrNull("one"));
        assertEquals(replacedTwo, collection.getOrNull("two"));
    }

    @Test
    public void replaceOneResourceByOtherRenameBoth() {
        var now = System.currentTimeMillis();
        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of());

        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        collection.put(init);
        collection.awaitActDone();
        assertEquals(init, collection.getOrNull("one"));

        var replace = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);
        collection.put(replace);
        collection.awaitActDone();
        assertEquals(replace, collection.getOrNull("two"));
        assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));

        var uniqueOne = new Resource(init).setName("unique-1").setUpdatedAt(now + 3);
        var uniqueTwo = new Resource(replace).setName("unique-2").setUpdatedAt(now + 3);
        collection.putAll(List.of(uniqueOne, uniqueTwo));
        collection.awaitActDone();
        assertEquals(uniqueOne, collection.getOrNull("one"));
        assertEquals(uniqueTwo, collection.getOrNull("two"));
    }

    @Test
    public void replaceResourceOnInit() {
        var now = System.currentTimeMillis();
        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        var replace = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);

        {
            var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(init, replace));
            assertEquals(replace, collection.getOrNull("two"));
            assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));
        }

        {
            var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(replace, init));
            assertEquals(replace, collection.getOrNull("two"));
            assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));
        }

        {
            var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(init, replace, init));
            assertEquals(replace, collection.getOrNull("two"));
            assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));
        }

        {
            var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(replace, init, replace));
            assertEquals(replace, collection.getOrNull("two"));
            assertEquals(new Resource(init).setReplaced(true), collection.getOrNull("one"));
        }
    }

    @Test
    public void revertReplacedResourceByDelete() {
        var now = System.currentTimeMillis();
        var init = resource("one").setName("not-unique-name").setUpdatedAt(now);
        var replace = resource("two").setName("not-unique-name").setUpdatedAt(now + 2);

        {
            var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(init, replace));
            collection.removeAll(List.of("one"));
            collection.awaitActDone();
            assertEquals(replace, collection.getOrNull("two"));
            assertNull(collection.getOrNull("one"));
        }

        {
            var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(init, replace));
            collection.removeAll(List.of("two"));
            collection.awaitActDone();
            assertEquals(init, collection.getOrNull("one"));
            assertNull(collection.getOrNull("two"));
        }
    }

    @Test
    public void interning() {
        var alice = resource("alice")
                .setName(new String("name".getBytes(StandardCharsets.UTF_8)))
                .setType(new String("type".getBytes(StandardCharsets.UTF_8)))
                .setService(new String("service".getBytes(StandardCharsets.UTF_8)))
                .setCloudId(new String("cloud".getBytes(StandardCharsets.UTF_8)))
                .setFolderId(new String("folder".getBytes(StandardCharsets.UTF_8)));

        var bob = resource("bob")
                .setName(new String("name".getBytes(StandardCharsets.UTF_8)))
                .setType(new String("type".getBytes(StandardCharsets.UTF_8)))
                .setService(new String("service".getBytes(StandardCharsets.UTF_8)))
                .setCloudId(new String("cloud".getBytes(StandardCharsets.UTF_8)))
                .setFolderId(new String("folder".getBytes(StandardCharsets.UTF_8)));

        var collection = new AwaitableResourcesCollection(executor, LEVELS_SIZES, List.of(alice));
        collection.put(bob);
        collection.awaitActDone();

        var internAlice = collection.getOrNull("alice");
        var internBob = collection.getOrNull("bob");

        assertSame(internAlice.cloudId, internBob.cloudId);
        assertSame(internAlice.folderId, internBob.folderId);
        assertSame(internAlice.service, internBob.service);
        assertSame(internAlice.type, internBob.type);
    }

    private static Resource resource(String resourceId) {
        return staticResource().setResourceId(resourceId).setName(resourceId);
    }
}
