package ru.yandex.solomon.name.resolver.logbroker;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import it.unimi.dsi.fastutil.longs.Long2ObjectArrayMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.persqueue.codec.Codec;
import ru.yandex.persqueue.codec.Encode;
import ru.yandex.persqueue.read.PartitionStream;
import ru.yandex.persqueue.read.PartitionStreamKey;

/**
 * @author Vladimir Gordiychuk
 */
public class PartitionStreamStub implements PartitionStream {
    private static final Logger logger = LoggerFactory.getLogger(PartitionStreamStub.class);
    private final PartitionStreamKey key;
    private final long id;
    private final Long2ObjectArrayMap<CountDownLatch> commitSyncs = new Long2ObjectArrayMap<>();

    public PartitionStreamStub(String topic, String cluster, long partition, long id) {
        this.key = new PartitionStreamKey(topic, cluster, partition);
        this.id = id;
    }

    @Override
    public PartitionStreamKey getKey() {
        return key;
    }

    @Override
    public long getAssignId() {
        return id;
    }

    @Override
    public String getTopicPath() {
        return key.topic;
    }

    @Override
    public String getClusterName() {
        return key.cluster;
    }

    @Override
    public long getPartition() {
        return key.partition;
    }

    @Override
    public void requestStatus() {
        // unsupported
    }

    public PartitionStreamDataEventStub dataMessage(String data) {
        Codec codec = ThreadLocalRandom.current().nextBoolean() ? Codec.RAW : Codec.GZIP;
        long offset = commitSyncs.size();
        commitSyncs.put(offset, new CountDownLatch(1));
        return new PartitionStreamDataEventStub(this, offset, codec, encode(codec, data));
    }

    @Override
    public synchronized void commit(long startOffset, long endOffset) {
        logger.debug("{} commit offset [{}, {})", this, startOffset, endOffset);
        for (long offset = startOffset; offset < endOffset; offset++) {
            var sync = commitSyncs.get(offset);
            if (sync == null) {
                throw new IllegalStateException("Unknown offset: " + offset);
            }
            if (sync.getCount() == 0) {
                throw new IllegalStateException("Already committed offset: " + offset);
            }
            sync.countDown();
        }
    }

    public synchronized CountDownLatch commitSync(long offset) {
        return commitSyncs.get(offset);
    }

    private static ByteString encode(Codec codec, String data) {
        var target = new ByteArrayOutputStream();
        try (var out = Encode.encodeStream(codec, target)) {
            out.write(data.getBytes(StandardCharsets.UTF_8));
            out.flush();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return UnsafeByteOperations.unsafeWrap(target.toByteArray());
    }

    @Override
    public String toString() {
        return "PartitionStream{" +
                "key=" + key +
                ", assignId=" + Long.toUnsignedString(id) +
                '}';
    }
}
