package ru.yandex.solomon.name.resolver.logbroker;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.SubmissionPublisher;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.persqueue.read.event.Event;
import ru.yandex.solomon.name.resolver.IssueTrackerNoop;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.sink.ResourceUpdateRequest;
import ru.yandex.solomon.name.resolver.sink.ResourceUpdaterStub;
import ru.yandex.solomon.util.ExceptionUtils;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;
import static ru.yandex.solomon.name.resolver.logbroker.ResourceFormat.resourceToJson;

/**
 * @author Vladimir Gordiychuk
 */
public class PersqueueSubscriberTest {
    private int MAX_UNCOMMITTED = 100;
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .build();

    private ResourceUpdaterStub updater;
    private Set<Resource> filtered;
    private PersqueueSubscriber subscriber;
    private SubmissionPublisher<Event> publisher;

    @Before
    public void setUp() {
        updater = new ResourceUpdaterStub();
        filtered = Sets.newConcurrentHashSet();
        var issueTracker = new IssueTrackerNoop();
        subscriber = new PersqueueSubscriber(updater, issueTracker, resource -> !filtered.contains(resource), MAX_UNCOMMITTED, ExceptionUtils::uncaughtException, new MetricRegistry());
        publisher = new SubmissionPublisher<>();
        publisher.subscribe(subscriber);
    }

    @After
    public void tearDown() {
        publisher.close();
    }

    @Test
    public void confirmAssignUnassign() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("topic", "cluster", 1, 3);
        var assign = new PartitionStreamCreateEventStub(partitionStream);
        publisher.submit(assign);
        assign.confirmSync.await();

        var unassign = new PartitionStreamDestroyEventStub(partitionStream);
        publisher.submit(unassign);
        unassign.confirmSync.await();
    }

    @Test
    public void commitNotValidMessages() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("topic", "cluster", 1, 3);
        publisher.submit(new PartitionStreamCreateEventStub(partitionStream));

        {
            var message = partitionStream.dataMessage("{invalid json");
            publisher.submit(message);
            message.commitSync().await();
        }

        {
            var message = partitionStream.dataMessage("invalid json}");
            publisher.submit(message);
            message.commitSync().await();
        }
    }

    @Test
    public void commitMessageOnlyWhenUpdaterConfirmIt() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("one", "man", 2, 1);
        publisher.submit(new PartitionStreamCreateEventStub(partitionStream));

        var resource = staticResource().setResourceComplexId(Map.of())
                .setResponsible("")
                .setEnvironment("")
                .setSeverity(Resource.Severity.UNKNOWN);
        var message = partitionStream.dataMessage(resourceToJson(resource));
        publisher.submit(message);

        var req = updater.requests.take();
        assertEquals(resource, req.resource());

        var commitSync = message.commitSync();
        assertFalse(commitSync.await(10, TimeUnit.MILLISECONDS));

        req.complete();
        commitSync.await();
    }

    @Test
    public void avoidSendInvalidMessage() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("one", "man", 2, 1);
        publisher.submit(new PartitionStreamCreateEventStub(partitionStream));

        var invalid = staticResource().setCloudId("").setResourceId("");
        var invalidMessage = partitionStream.dataMessage(resourceToJson(invalid));
        publisher.submit(invalidMessage);

        var valid = staticResource().setResourceComplexId(Map.of())
                .setResponsible("")
                .setEnvironment("")
                .setSeverity(Resource.Severity.UNKNOWN);
        var validMessage = partitionStream.dataMessage(resourceToJson(valid));
        publisher.submit(validMessage);

        var req = updater.requests.take();
        assertEquals(valid, req.resource());
        req.complete();

        var invalidSync = invalidMessage.commitSync();
        invalidSync.await();

        var validSync = validMessage.commitSync();
        validSync.await();
    }

    @Test
    public void avoidSendFilteredMessage() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("one", "man", 2, 1);
        publisher.submit(new PartitionStreamCreateEventStub(partitionStream));

        var alice = staticResource().setResourceId("alice").setResourceComplexId(Map.of())
                .setResponsible("")
                .setEnvironment("")
                .setSeverity(Resource.Severity.UNKNOWN);
        var aliceMessage = partitionStream.dataMessage(resourceToJson(alice));
        this.filtered.add(alice);
        publisher.submit(aliceMessage);

        var bob = staticResource().setResourceId("bob").setResourceComplexId(Map.of())
                .setResponsible("")
                .setEnvironment("")
                .setSeverity(Resource.Severity.UNKNOWN);
        var bobMessage = partitionStream.dataMessage(resourceToJson(bob));
        publisher.submit(bobMessage);

        var req = updater.requests.take();
        assertEquals(bob, req.resource());
        req.complete();

        aliceMessage.commitSync().await();
        bobMessage.commitSync().await();
    }

    @Test
    public void parsedEventDeliverToUpdater() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("topic", "cluster", 1, 42);
        publisher.submit(new PartitionStreamCreateEventStub(partitionStream));

        List<Resource> expectedResources = new ArrayList<>();
        for (int index = 0; index < MAX_UNCOMMITTED; index++) {
            var resource = staticResource().setResourceId(Integer.toString(index)).setResourceComplexId(Map.of())
                    .setResponsible("")
                    .setEnvironment("")
                    .setSeverity(Resource.Severity.UNKNOWN);
            expectedResources.add(resource);
            publisher.submit(partitionStream.dataMessage(resourceToJson(resource)));
        }

        List<Resource> actualResource = new ArrayList<>();
        while (actualResource.size() != expectedResources.size()) {
            var req = updater.requests.take();
            actualResource.add(req.resource());
            req.complete();
        }

        expectedResources.sort(Comparator.comparing(o -> o.resourceId));
        actualResource.sort(Comparator.comparing(o -> o.resourceId));
        assertArrayEquals(expectedResources.toArray(), actualResource.toArray());
    }

    @Test
    public void limitedUncommittedMessage() throws InterruptedException {
        var partitionStream = new PartitionStreamStub("topic", "cluster", 1, 42);
        publisher.submit(new PartitionStreamCreateEventStub(partitionStream));

        int totalMessage = MAX_UNCOMMITTED * 2;
        List<Resource> expectedResources = new ArrayList<>();
        for (int index = 0; index < totalMessage; index++) {
            var resource = staticResource().setResourceId(Integer.toString(index)).setResourceComplexId(Map.of())
                    .setResponsible("")
                    .setEnvironment("")
                    .setSeverity(Resource.Severity.UNKNOWN);
            expectedResources.add(resource);
            publisher.submit(partitionStream.dataMessage(resourceToJson(resource)));
        }

        List<ResourceUpdateRequest> requests = new ArrayList<>();
        for (int index = 0; index < MAX_UNCOMMITTED; index++) {
            requests.add(updater.requests.take());
        }

        assertNull(updater.requests.poll(100, TimeUnit.MILLISECONDS));
        for (int index = MAX_UNCOMMITTED; index < MAX_UNCOMMITTED * 2; index++) {
            requests.get(index - MAX_UNCOMMITTED).complete();
            requests.add(updater.requests.take());
        }

        var actual = requests.stream()
                .map(ResourceUpdateRequest::resource)
                .collect(Collectors.toList());
        expectedResources.sort(Comparator.comparing(o -> o.resourceId));
        actual.sort(Comparator.comparing(o -> o.resourceId));
        assertArrayEquals(expectedResources.toArray(), actual.toArray());
    }
}
