package ru.yandex.solomon.name.resolver.logbroker;

import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.core.db.model.ReferenceConf;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceFilterTest {
    private ResourceFilter filter;

    @Before
    public void setUp() throws Exception {
        filter = new ResourceFilter(true);
    }

    @Test
    public void absentServiceProviders() {
        filter.onUpdateServiceProviders(Map.of());
        assertFalse(filter.test(staticResource()));
    }

    @Test
    public void absentFields() {
        filter.onUpdateServiceProviders(Map.of());
        assertFalse(filter.test(new Resource()
                .setCloudId("cloudId")
                .setService("service")
                .setResourceId("resourceId")
                .setResourceComplexId(Map.of("k1", "v1", "k2", "v2"))
                .setUpdatedAt(System.currentTimeMillis())
                .setDeletedAt(0)));
    }

    @Test
    public void filterNotMatchingByReference() {
        var serviceProvider = ServiceProvider.newBuilder()
                .setId("compute")
                .setReferences(List.of(new ReferenceConf("resource_id", List.of("compute"), List.of("disk", "vm"), false)))
                .build();
        filter.onUpdateServiceProviders(Map.of("compute", serviceProvider));
        assertTrue(filter.test(staticResource().setService("compute").setType("disk")));
        assertTrue(filter.test(staticResource().setService("compute").setType("vm")));
        assertFalse(filter.test(staticResource().setService("compute").setType("tmp")));
        assertFalse(filter.test(staticResource().setService("certificate-manager").setType("vm")));
    }

    @Test
    public void filterNotMatchingByMultipleReference() {
        var compute = ServiceProvider.newBuilder()
                .setId("compute")
                .setReferences(List.of(
                        new ReferenceConf("resource_id", List.of("compute"), List.of("vm"), false),
                        new ReferenceConf("disk", List.of("compute"), List.of("disk"), false)
                ))
                .build();
        var certificateManager = ServiceProvider.newBuilder()
                .setId("certificate-manager")
                .setReferences(List.of(new ReferenceConf("certificate", List.of("certificate-manager"), List.of("certificate"), false)))
                .build();
        filter.onUpdateServiceProviders(Map.of("compute", compute, "certificate-manager", certificateManager));
        assertTrue(filter.test(staticResource().setService("compute").setType("disk")));
        assertTrue(filter.test(staticResource().setService("compute").setType("vm")));
        assertTrue(filter.test(staticResource().setService("certificate-manager").setType("certificate")));
        assertFalse(filter.test(staticResource().setService("compute").setType("tmp")));
        assertFalse(filter.test(staticResource().setService("certificate-manager").setType("vm")));
        assertFalse(filter.test(staticResource().setService("compute").setType("certificate")));
    }

    @Test
    public void disableFilter() {
        filter = new ResourceFilter(false);
        filter.onUpdateServiceProviders(Map.of());
        assertTrue(filter.test(staticResource()));
    }

    @Test
    public void readyFlag() {
        assertFalse(filter.isReady());
        filter.onUpdateServiceProviders(Map.of());
        assertTrue(filter.isReady());
    }
}
