package ru.yandex.solomon.name.resolver.logbroker;

import java.io.IOException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Map;

import org.junit.Test;

import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;
import static ru.yandex.solomon.name.resolver.logbroker.ResourceFormat.resourceToJson;

/**
 * @author Vladimir Gordiychuk
 */
public class ResourceParserTest {

    @Test
    public void parseDeleted() throws IOException {
        String json = """
                {
                    "service":"compute",
                    "cloud_id":"aoe96r0tq4cq6gf6i2no",
                    "folder_id":"aoe9uqbo3c7i8f26fcke",
                    "resource_id":"c8r1htbb555sckd2frva",
                    "resource_type":"disk",
                    "permission":"compute.disks.get",
                    "timestamp":"2020-11-25T16:18:14.476201+00:00",
                    "deleted":"2020-11-25T16:18:14.476201+00:00",
                    "attributes":{}
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("compute", resource.service);
        assertEquals("aoe96r0tq4cq6gf6i2no", resource.cloudId);
        assertEquals("aoe9uqbo3c7i8f26fcke", resource.folderId);
        assertEquals("c8r1htbb555sckd2frva", resource.resourceId);
        assertEquals("disk", resource.type);
        assertEquals("timestamp", Instant.parse("2020-11-25T16:18:14.476201+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", Instant.parse("2020-11-25T16:18:14.476201+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.deletedAt));
        assertEquals("", resource.name);
    }

    @Test
    public void parseDeletedComputeInstance() throws IOException {
        String json = """
                {
                    "deleted":"2020-12-28T14:41:59.216246+00:00",
                    "service":"compute",
                    "resource_type":"instance",
                    "resource_id":"d9h429lu97ub8s5p6g45",
                    "permission":"compute.instances.get",
                    "attributes":{},
                    "cloud_id":"aoeh8qdokbugui4tliup",
                    "folder_id":"aoeg1t1sj8v6ni4g6vqg",
                    "timestamp":"2020-12-28T14:41:59.216246+00:00"
                }
                """;
        var resource = ResourceParser.parse(json);
        assertEquals("compute", resource.service);
        assertEquals("aoeh8qdokbugui4tliup", resource.cloudId);
        assertEquals("aoeg1t1sj8v6ni4g6vqg", resource.folderId);
        assertEquals("d9h429lu97ub8s5p6g45", resource.resourceId);
        assertEquals("instance", resource.type);
        assertEquals("timestamp", Instant.parse("2020-12-28T14:41:59.216246+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", Instant.parse("2020-12-28T14:41:59.216246+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.deletedAt));
        assertEquals("", resource.name);
    }

    @Test
    public void parseComputeInstance() throws IOException {
        String json = """
                {
                    "service":"compute",
                    "cloud_id":"aoenio4bf5tauqgr4tj4",
                    "folder_id":"aoec3lfd01kg3ea3kan0",
                    "resource_id":"d9hqv7e10l3g51vqgi57",
                    "resource_type":"instance",
                    "permission":"compute.instances.get",
                    "timestamp":"2020-10-27T08:06:43.340107+00:00",
                    "deleted":"",
                    "attributes":{
                        "description":"",
                        "fqdn":"amcqf4sijdj2h2shdcrn-ojuk.ru-central1.internal",
                        "hostname":"amcqf4sijdj2h2shdcrn-ojuk",
                        "name":"amcqf4sijdj2h2shdcrn-ojuk"
                    }
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("compute", resource.service);
        assertEquals("aoenio4bf5tauqgr4tj4", resource.cloudId);
        assertEquals("aoec3lfd01kg3ea3kan0", resource.folderId);
        assertEquals("d9hqv7e10l3g51vqgi57", resource.resourceId);
        assertEquals("instance", resource.type);
        assertEquals("timestamp", Instant.parse("2020-10-27T08:06:43.340107+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("amcqf4sijdj2h2shdcrn-ojuk", resource.name);
    }

    @Test
    public void parseComputeDisk() throws IOException {
        String json = """
                {
                    "service":"compute",
                    "cloud_id":"aoe96r0tq4cq6gf6i2no",
                    "folder_id":"aoe9uqbo3c7i8f26fcke",
                    "resource_id":"d9hvo32hnrfuhi7f3cih",
                    "resource_type":"disk",
                    "permission":"compute.disks.get",
                    "timestamp":"2020-10-27T03:26:37.907109+00:00",
                    "deleted":"",
                    "attributes":{
                        "description":"",
                        "instance_ids":[],
                        "name":"k8s-csi-37a4d2c340bf13b89d7b08ef70902d86c72df046"
                    }
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("compute", resource.service);
        assertEquals("aoe96r0tq4cq6gf6i2no", resource.cloudId);
        assertEquals("aoe9uqbo3c7i8f26fcke", resource.folderId);
        assertEquals("d9hvo32hnrfuhi7f3cih", resource.resourceId);
        assertEquals("disk", resource.type);
        assertEquals("timestamp", Instant.parse("2020-10-27T03:26:37.907109+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("k8s-csi-37a4d2c340bf13b89d7b08ef70902d86c72df046", resource.name);
    }

    @Test
    public void parseManagedSqlServer() throws IOException {
        String json = """
                {
                    "deleted": "",
                    "service": "managed-sqlserver",
                    "cloud_id": "aoe9shbqc2v314v7fp3d",
                    "folder_id": "aoed5i52uquf5jio0oec",
                    "timestamp": "2020-10-27T06:56:46Z",
                    "attributes": {
                        "name": "dbaas_e2e_compute_preprod",
                        "hosts": ["rc1a-p3uh93an5r.mdb.cloud-preprod.yandex.net"],
                        "users": ["alice"],
                        "labels": {},
                        "databases": ["dbaas_e2e_compute_preprod"],
                        "description": ""
                    },
                    "permission": "mdb.all.read",
                    "resource_id": "e4ussdm7676u11jva4tr",
                    "resource_type": "cluster"
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("managed-sqlserver", resource.service);
        assertEquals("aoe9shbqc2v314v7fp3d", resource.cloudId);
        assertEquals("aoed5i52uquf5jio0oec", resource.folderId);
        assertEquals("e4ussdm7676u11jva4tr", resource.resourceId);
        assertEquals("cluster", resource.type);
        assertEquals("timestamp", Instant.parse("2020-10-27T06:56:46Z"), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("dbaas_e2e_compute_preprod", resource.name);
    }

    @Test
    public void parseManagedPostgresql() throws IOException {
        String json = """
                {
                    "deleted": "",
                    "service": "managed-postgresql",
                    "cloud_id": "aoe9shbqc2v314v7fp3d",
                    "folder_id": "aoed5i52uquf5jio0oec",
                    "timestamp": "2020-10-27T00:03:24.507838+00:00",
                    "attributes": {
                        "name": "dbaas_e2e_compute_preprod",
                        "hosts": [
                            "rc1a-mj46mluk92zyqfux.mdb.cloud-preprod.yandex.net",
                            "rc1b-zpf8y9ya4q3w4qum.mdb.cloud-preprod.yandex.net",
                            "rc1c-j1pacdu9l20oir32.mdb.cloud-preprod.yandex.net"
                        ],
                        "users": ["alice"],
                        "labels": {},
                        "databases": ["dbaas_e2e_compute_preprod"],
                        "description": ""
                    },
                    "permission": "mdb.all.read",
                    "resource_id": "e4unmhv872de4jbmj7rs",
                    "resource_type": "cluster"
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("managed-postgresql", resource.service);
        assertEquals("aoe9shbqc2v314v7fp3d", resource.cloudId);
        assertEquals("aoed5i52uquf5jio0oec", resource.folderId);
        assertEquals("e4unmhv872de4jbmj7rs", resource.resourceId);
        assertEquals("cluster", resource.type);
        assertEquals("timestamp", Instant.parse("2020-10-27T00:03:24.507838+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("dbaas_e2e_compute_preprod", resource.name);
    }

    @Test
    public void parseTestJson() throws IOException {
        var expected = staticResource().setResourceComplexId(Map.of())
                .setResponsible("")
                .setEnvironment("")
                .setSeverity(Resource.Severity.UNKNOWN);
        String json = resourceToJson(expected);
        var actual = ResourceParser.parse(json);
        assertEquals(expected, actual);
    }

    @Test
    public void parseNameOnTopLevel() throws IOException {
        String json = """
                {
                    "service":"compute",
                    "cloud_id":"aoenio4bf5tauqgr4tj4",
                    "folder_id":"aoec3lfd01kg3ea3kan0",
                    "resource_id":"d9hqv7e10l3g51vqgi57",
                    "resource_type":"instance",
                    "permission":"compute.instances.get",
                    "timestamp":"2020-10-27T08:06:43.340107+00:00",
                    "deleted":"",
                    "name": "amcqf4sijdj2h2shdcrn-ojuk",
                    "attributes":{
                        "description":"",
                        "fqdn":"amcqf4sijdj2h2shdcrn-ojuk.ru-central1.internal",
                        "hostname":"amcqf4sijdj2h2shdcrn-ojuk"
                    }
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("compute", resource.service);
        assertEquals("aoenio4bf5tauqgr4tj4", resource.cloudId);
        assertEquals("aoec3lfd01kg3ea3kan0", resource.folderId);
        assertEquals("d9hqv7e10l3g51vqgi57", resource.resourceId);
        assertEquals("instance", resource.type);
        assertEquals("timestamp", Instant.parse("2020-10-27T08:06:43.340107+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("amcqf4sijdj2h2shdcrn-ojuk", resource.name);
    }

    @Test
    public void parseBatchResources() throws IOException {
        String json = """
                {
                    "service": "compute",
                    "cloud_id": "aoeh8qdokbugui4tliup",
                    "folder_id": "aoeg1t1sj8v6ni4g6vqg",
                    "resource_id": "d9h8mq4sttom7hgu9v94",
                    "resource_type": "instance",
                    "permission": "compute.instances.get",
                    "timestamp": "2020-12-28T15:51:02.277664+00:00",
                    "deleted": "",
                    "attributes": {
                        "description": "",
                        "fqdn": "d9h8mq4sttom7hgu9v94.auto.internal",
                        "hostname": "",
                        "name": "vm-e2e-ru-central1-c-network-nofip"
                    }
                }
                {
                    "service": "compute",
                    "cloud_id": "aoeh8qdokbugui4tliup",
                    "folder_id": "aoeg1t1sj8v6ni4g6vqg",
                    "resource_id": "d9h8mq4sttom7hgu9v94",
                    "resource_type": "instance",
                    "permission": "compute.instances.get",
                    "timestamp": "2020-12-28T15:52:32.260827+00:00",
                    "deleted": "2020-12-28T15:52:32.260827+00:00",
                    "attributes": {}
                }
                """;

        try (var parser = new ResourceParser(json)) {
            {
                assertTrue(parser.hasNext());
                var resource = parser.next();
                assertEquals("compute", resource.service);
                assertEquals("aoeh8qdokbugui4tliup", resource.cloudId);
                assertEquals("aoeg1t1sj8v6ni4g6vqg", resource.folderId);
                assertEquals("d9h8mq4sttom7hgu9v94", resource.resourceId);
                assertEquals("instance", resource.type);
                assertEquals("timestamp", Instant.parse("2020-12-28T15:51:02.277664+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
                assertEquals("deleted", 0, resource.deletedAt);
                assertEquals("vm-e2e-ru-central1-c-network-nofip", resource.name);
            }
            {
                assertTrue(parser.hasNext());
                var resource = parser.next();
                assertEquals("compute", resource.service);
                assertEquals("aoeh8qdokbugui4tliup", resource.cloudId);
                assertEquals("aoeg1t1sj8v6ni4g6vqg", resource.folderId);
                assertEquals("d9h8mq4sttom7hgu9v94", resource.resourceId);
                assertEquals("instance", resource.type);
                assertEquals("timestamp", Instant.parse("2020-12-28T15:52:32.260827+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
                assertEquals("deleted", Instant.parse("2020-12-28T15:52:32.260827+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.deletedAt));
                assertEquals("", resource.name);
            }
            assertFalse(parser.hasNext());
        }
    }

    @Test
    public void timestampCanBeNull() throws IOException {
        String json = """
                {
                    "resource_type":"dns",
                    "timestamp":"2021-02-17T10:42:30.258Z",
                    "resource_id":"aet1ue860hfh32hp26t4",
                    "name":"auto-c64ocqfeu80bgcb72u1o-168_192_in-addr_arpa_",
                    "service":"zone",
                    "deleted":null,
                    "permission":"dns.zones.get",
                    "cloud_id":"aoeh8qdokbugui4tliup",
                    "folder_id":"aoeg1t1sj8v6ni4g6vqg",
                    "attributes":{
                        "zone":"168.192.in-addr.arpa."
                    },
                    "reindex_timestamp":null,
                    "resource_path":[
                        {
                            "resource_type":"resource-manager.cloud",
                            "resource_id":"aoeh8qdokbugui4tliup"
                        },
                        {
                            "resource_type":"resource-manager.folder",
                            "resource_id":"aoeg1t1sj8v6ni4g6vqg"
                        },
                        {
                            "resource_type":"dns.zone",
                            "resource_id":"aet1ue860hfh32hp26t4"
                        }
                    ]
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("zone", resource.service);
        assertEquals("aoeh8qdokbugui4tliup", resource.cloudId);
        assertEquals("aoeg1t1sj8v6ni4g6vqg", resource.folderId);
        assertEquals("aet1ue860hfh32hp26t4", resource.resourceId);
        assertEquals("dns", resource.type);
        assertEquals("timestamp", Instant.parse("2021-02-17T10:42:30.258Z").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("reindex_timestamp", 0, resource.reindexAt);
        assertEquals("auto-c64ocqfeu80bgcb72u1o-168_192_in-addr_arpa_", resource.name);
    }

    @Test
    public void parseReindexTimestamp() throws IOException {
        String json = """
                {
                    "resource_type":"dns",
                    "timestamp":"2021-02-17T11:13:14.179Z",
                    "resource_id":"aet4541g63acv1ecmkgk",
                    "name":"auto-c64dutioramq7a0r9h3f-_",
                    "service":"zone",
                    "permission":"dns.zones.get",
                    "cloud_id":"aoeh8qdokbugui4tliup",
                    "folder_id":"aoeg1t1sj8v6ni4g6vqg",
                    "reindex_timestamp":"2021-02-20T11:00:30Z",
                    "resource_path":[
                        {
                            "resource_type":"resource-manager.cloud",
                            "resource_id":"aoeh8qdokbugui4tliup"
                        },
                        {
                            "resource_type":"resource-manager.folder",
                            "resource_id":"aoeg1t1sj8v6ni4g6vqg"
                        },
                        {
                            "resource_type":"dns.zone",
                            "resource_id":"aet4541g63acv1ecmkgk"
                        }
                    ]
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("zone", resource.service);
        assertEquals("aoeh8qdokbugui4tliup", resource.cloudId);
        assertEquals("aoeg1t1sj8v6ni4g6vqg", resource.folderId);
        assertEquals("aet4541g63acv1ecmkgk", resource.resourceId);
        assertEquals("dns", resource.type);
        assertEquals("timestamp", Instant.parse("2021-02-17T11:13:14.179Z").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("auto-c64dutioramq7a0r9h3f-_", resource.name);
        assertEquals("reindex_timestamp", Instant.parse("2021-02-20T11:00:30Z").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.reindexAt));
    }

    @Test
    public void parseNullAttribute() throws IOException {
        String json = """
                {
                    "service": "mks",
                    "cloud_id": "aoe96r0tq4cq6gf6i2no",
                    "folder_id": "aoe94u7icen30rcb2eup",
                    "resource_id": "c495gnc9lsasmsnvuahh",
                    "resource_type": "cluster",
                    "resource_path": [
                        {
                            "resource-manager.folder": "aoe94u7icen30rcb2eup"
                        }
                    ],
                    "name": "e2e",
                    "permission": "managed-kubernetes.clusters.get",
                    "timestamp": "2021-06-16T14:19:56.422265+00:00",
                    "reindex_timestamp": "",
                    "deleted": "",
                    "attributes": null
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("mks", resource.service);
        assertEquals("aoe96r0tq4cq6gf6i2no", resource.cloudId);
        assertEquals("aoe94u7icen30rcb2eup", resource.folderId);
        assertEquals("c495gnc9lsasmsnvuahh", resource.resourceId);
        assertEquals("cluster", resource.type);
        assertEquals("timestamp", Instant.parse("2021-06-16T14:19:56.422265+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("e2e", resource.name);
        assertEquals("reindex_timestamp", 0, resource.reindexAt);
    }

    @Test
    public void emptyNameNullAttributes() throws IOException {
        var json = """
                {
                    "service": "mks",
                    "cloud_id": "aoe96r0tq4cq6gf6i2no",
                    "folder_id": "aoeeqkne82acmg0n95k1",
                    "resource_id": "c49kmktgkl8q5f95kjcn",
                    "resource_type": "cluster",
                    "resource_path": [
                        {
                            "resource-manager.folder": "aoeeqkne82acmg0n95k1"
                        }
                    ],
                    "name": "",
                    "permission": "managed-kubernetes.clusters.get",
                    "timestamp": "2021-06-16T11:51:56.189593+00:00",
                    "reindex_timestamp": "2021-06-16T11:51:56.189593+00:00",
                    "deleted": "",
                    "attributes": null
                }
                """;

        var resource = ResourceParser.parse(json);
        assertEquals("mks", resource.service);
        assertEquals("aoe96r0tq4cq6gf6i2no", resource.cloudId);
        assertEquals("aoeeqkne82acmg0n95k1", resource.folderId);
        assertEquals("c49kmktgkl8q5f95kjcn", resource.resourceId);
        assertEquals("cluster", resource.type);
        assertEquals("timestamp", Instant.parse("2021-06-16T11:51:56.189593+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.updatedAt));
        assertEquals("deleted", 0, resource.deletedAt);
        assertEquals("", resource.name);
        assertEquals("reindex_timestamp", Instant.parse("2021-06-16T11:51:56.189593+00:00").truncatedTo(ChronoUnit.MILLIS), Instant.ofEpochMilli(resource.reindexAt));
    }
}
