package ru.yandex.solomon.name.resolver.sink;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

import io.grpc.Status;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardSinkClientStub implements ShardSinkClient {
    private ConcurrentHashMap<String, Shard> shardById = new ConcurrentHashMap<>();
    public volatile RuntimeException error;

    public void assignShardToNode(String shardId, String node) {
        var shard = getOrCreateShard(shardId);
        shard.node.set(node);
    }

    public List<Resource> takeResources(String shardId) {
        return getOrCreateShard(shardId).resources.dequeueAll();
    }

    @Override
    public CompletableFuture<String> resolveNode(String shardId) {
        return CompletableFuture.supplyAsync(() -> {
            ensureError();
            var shard = getOrCreateShard(shardId);
            return shard.node.get();
        });
    }

    @Override
    public CompletableFuture<Void> update(String node, String shardId, List<Resource> resources, boolean removeOtherOfService, String serviceProviderId) {
        return CompletableFuture.runAsync(() -> {
            ensureError();
            var shard = getOrCreateShard(shardId);
            if (!Objects.equals(node, shard.node.get())) {
                throw Status.NOT_FOUND
                        .withDescription("shard located on another node")
                        .asRuntimeException();
            }
            shard.resources.enqueueAll(resources);
        });
    }

    private void ensureError() {
        var copy = error;
        if (copy != null) {
            throw copy;
        }
    }

    private Shard getOrCreateShard(String shardId) {
        return shardById.computeIfAbsent(shardId, Shard::new);
    }

    private static class Shard {
        private String shardId;
        private AtomicReference<String> node = new AtomicReference<>();
        private ArrayListLockQueue<Resource> resources = new ArrayListLockQueue<>();

        public Shard(String shardId) {
            this.shardId = shardId;
        }
    }
}
