package ru.yandex.solomon.name.resolver.sink;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.junit.Assert.assertArrayEquals;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardSinkDispatcherTest {

    @Rule
    public TestName testName = new TestName();
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private ShardSinkClientStub client;
    private String shardId;
    private ShardSinkDispatcher dispatcher;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        client = new ShardSinkClientStub();
        shardId = testName.getMethodName();
        dispatcher = new ShardSinkDispatcher(client, ForkJoinPool.commonPool(), timer, new ShardSinkMetrics(new MetricRegistry()));
    }

    @After
    public void tearDown() {
        timer.shutdownNow();
    }

    @Test
    public void updateSameShard() throws InterruptedException {
        String shardId = "myShard";
        List<Resource> expected = new ArrayList<>();
        client.assignShardToNode(shardId, "myNode");
        CountDownLatch sync = new CountDownLatch(10);
        for (int index = 0; index < 10; index++) {
            var resource = staticResource()
                    .setCloudId(shardId)
                    .setResourceId("resource-" + index);
            write(resource, sync);
            expected.add(resource);
        }

        sync.await();
        assertArrayEquals(expected.toArray(), client.takeResources(shardId).toArray());
    }

    @Test
    public void updateDifferentShards() throws InterruptedException {
        List<Resource> alice = new ArrayList<>();
        List<Resource> bob = new ArrayList<>();
        client.assignShardToNode("myShard-0", "alice");
        client.assignShardToNode("myShard-1", "bob");
        CountDownLatch sync = new CountDownLatch(10);
        for (int index = 0; index < 10; index++) {
            if ((index % 2) == 0) {
                var resource = staticResource()
                        .setCloudId("myShard-0")
                        .setResourceId("resource-" + index);
                write(resource, sync);
                alice.add(resource);
            } else {
                var resource = staticResource()
                        .setCloudId("myShard-1")
                        .setResourceId("resource-" + index);
                write(resource, sync);
                bob.add(resource);
            }
        }

        sync.await();
        assertArrayEquals(alice.toArray(), client.takeResources("myShard-0").toArray());
        assertArrayEquals(bob.toArray(), client.takeResources("myShard-1").toArray());
    }

    private void write(Resource resource, CountDownLatch sync) {
        dispatcher.update(new UpdateRequestStub(resource, sync));
    }
}
