package ru.yandex.solomon.name.resolver.sink;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import io.grpc.Status;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardSinkTest {

    @Rule
    public TestName testName = new TestName();
    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private ShardSinkClientStub client;
    private String shardId;
    private ShardSink sink;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        client = new ShardSinkClientStub();
        shardId = testName.getMethodName();
        sink = new ShardSink(shardId, client, ForkJoinPool.commonPool(), timer, new ShardSinkMetrics(new MetricRegistry()));
    }

    @After
    public void tearDown() {
        timer.shutdownNow();
    }

    @Test
    public void sequentialUpdateResource() throws InterruptedException {
        client.assignShardToNode(shardId, "alice");
        for (int index = 0; index < 10; index++) {
            var sync = new CountDownLatch(1);
            var resource = staticResource()
                    .setCloudId(shardId)
                    .setResourceId("resource-id-" + index)
                    .setName("name-" + index);
            write(resource, sync);
            sync.await();
            assertEquals(List.of(resource), client.takeResources(shardId));
        }
    }

    @Test
    public void multipleUpdateResource() throws InterruptedException {
        client.assignShardToNode(shardId, "alice");
        List<Resource> expected = new ArrayList<>(1000);
        CountDownLatch sync = new CountDownLatch(1000);
        for (int index = 0; index < 1000; index++) {
            var resource = staticResource()
                    .setCloudId(shardId)
                    .setResourceId("resource-id-" + index)
                    .setName("name-" + index);
            expected.add(resource);
            write(resource, sync);
        }
        sync.await();
        assertArrayEquals(expected.toArray(), client.takeResources(shardId).toArray());
    }

    @Test
    public void shardMoveToAnotherHostAtTheMiddle() throws InterruptedException {
        client.assignShardToNode(shardId, "alice");
        int count = 1000;
        List<Resource> expected = new ArrayList<>(count);
        CountDownLatch sync = new CountDownLatch(count);
        for (int index = 0; index < count; index++) {
            var resource = staticResource()
                    .setCloudId(shardId)
                    .setResourceId("resource-id-" + index)
                    .setName("name-" + index);
            expected.add(resource);
            write(resource, sync);
            if (index == count / 2) {
                client.assignShardToNode(shardId, "bob");
            }
        }
        sync.await();
        assertArrayEquals(expected.toArray(), client.takeResources(shardId).toArray());
    }

    @Test
    public void shardMoveToAnotherHost() throws InterruptedException {
        for (int index = 0; index < 10; index++) {
            client.assignShardToNode(shardId, "node-" + index);
            var sync = new CountDownLatch(1);
            var resource = staticResource()
                    .setCloudId(shardId)
                    .setResourceId("resource-1");
            write(resource, sync);
            sync.await();
            assertEquals(List.of(resource), client.takeResources(shardId));
        }
    }

    @Test
    public void retryResolveShardAssignment() throws InterruptedException {
        client.error = Status.UNAVAILABLE.asRuntimeException();
        client.assignShardToNode(shardId, "alice");
        var sync = new CountDownLatch(1);
        var resource = staticResource().setCloudId(shardId);
        write(resource, sync);

        assertFalse(sync.await(100, TimeUnit.MILLISECONDS));
        client.error = null;
        sync.await();
        assertEquals(List.of(resource), client.takeResources(shardId));
    }

    private void write(Resource resource, CountDownLatch sync) {
        sink.update(new UpdateRequestStub(resource, sync));
    }
}
