package ru.yandex.solomon.name.resolver.ttl;

import java.util.Map;

import it.unimi.dsi.fastutil.objects.Object2LongMap;
import it.unimi.dsi.fastutil.objects.Object2LongMaps;
import org.junit.Test;

import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.logbroker.ResourceValidator;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class TtlPredicateTest {

    @Test
    public void unknownServiceResourceDeleted() {
        long now = System.currentTimeMillis();
        var remove = resource(now).setDeletedAt(now - 1_000);
        var noRemove = resource(now).setDeletedAt(now - 10);
        var predicate = predicate(now, 100, Object2LongMaps.emptyMap());
        assertTrue(predicate.test(remove));
        assertFalse(predicate.test(noRemove));
    }

    @Test
    public void unknownServiceResourceDeletedReplaced() {
        long now = System.currentTimeMillis();
        var remove = resource(now).setDeletedAt(now - 1_000).setReplaced(true);
        var noRemove = resource(now).setDeletedAt(now - 10).setReplaced(true);
        var predicate = predicate(now, 100, Object2LongMaps.emptyMap());
        assertTrue(predicate.test(remove));
        assertFalse(predicate.test(noRemove));
    }

    @Test
    public void unknownServiceResourceAlive() {
        long now = System.currentTimeMillis();
        var noRemove = resource(now);
        var predicate = predicate(now, 100, Object2LongMaps.emptyMap());
        assertFalse(predicate.test(noRemove));
    }

    @Test
    public void unknownServiceResourceAliveReplaced() {
        long now = System.currentTimeMillis();
        var noRemove = resource(now).setReplaced(true);
        var predicate = predicate(now, 100, Object2LongMaps.emptyMap());
        assertFalse(predicate.test(noRemove));
    }

    @Test
    public void knownServiceResourceDeleted() {
        long now = System.currentTimeMillis();
        var predicate = predicate(now, 100, Object2LongMaps.singleton("compute", 5_000));
        assertTrue(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 10_000)));
        assertTrue(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 6_000)));
        assertTrue(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 5_000 - 100)));
        assertFalse(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 5_000)));
        assertFalse(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now)));
    }

    @Test
    public void knownServiceResourceDeletedReplaced() {
        long now = System.currentTimeMillis();
        var predicate = predicate(now, 100, Object2LongMaps.singleton("compute", 5_000));
        assertTrue(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 10_000).setReplaced(true)));
        assertTrue(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 6_000).setReplaced(true)));
        assertTrue(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 100).setReplaced(true)));
        assertFalse(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now - 80).setReplaced(true)));
        assertFalse(predicate.test(resource(now).setService("compute").setUpdatedAt(now).setDeletedAt(now).setReplaced(true)));
    }

    @Test
    public void knowServiceResourceAlive() {
        long now = System.currentTimeMillis();
        var predicate = predicate(now, 100, Object2LongMaps.singleton("compute", 5_000));
        assertFalse(predicate.test(resource(now).setService("compute")));
        assertFalse(predicate.test(resource(now).setService("compute").setReplaced(true)));
    }

    @Test
    public void knownServiceDisableTtl() {
        long now = System.currentTimeMillis();
        var predicate = predicate(now, 100, Object2LongMaps.singleton("compute", 0));
        for (long deleteTime : new long[]{10_000, 5_000, 100, 50, 0}) {
            var deleted = resource(now).setService("compute").setDeletedAt(now - deleteTime);
            assertFalse(deleted.toString(), predicate.test(deleted));

            var deletedReplaced = resource(now).setService("compute").setDeletedAt(now - deleteTime).setReplaced(true);
            assertFalse(deletedReplaced.toString(), predicate.test(deletedReplaced));

            var alive = resource(now).setService("compute").setUpdatedAt(now - deleteTime);
            assertFalse(alive.toString(), predicate.test(alive));

            var aliveReplaced = resource(now).setService("compute").setUpdatedAt(now - deleteTime).setReplaced(true);
            assertFalse(aliveReplaced.toString(), predicate.test(aliveReplaced));
        }
    }

    @Test
    public void invalidResource() {
        long now = System.currentTimeMillis();
        var remove = resource(now).setCloudId("").setResourceId("");
        var noRemove = resource(now);
        var predicate = predicate(now, 100, Object2LongMaps.emptyMap());
        assertTrue(predicate.test(remove));
        assertFalse(predicate.test(noRemove));
    }

    @Test
    public void absentFields() {
        long now = System.currentTimeMillis();
        var predicate = new TtlPredicate(now, 100, Object2LongMaps.singleton("compute", 5_000), ResourceValidator.LIGHT_VALIDATOR);
        assertFalse(predicate.test(new Resource()
                .setCloudId("cloudId")
                .setService("compute")
                .setResourceId("resourceId")
                .setResourceComplexId(Map.of("k1", "v1", "k2", "v2"))
                .setUpdatedAt(System.currentTimeMillis())
                .setDeletedAt(0)));
        assertTrue(predicate.test(new Resource()
                .setCloudId("cloudId")
                .setService("compute")
                .setResourceId("resourceId")
                .setResourceComplexId(Map.of("k1", "v1", "k2", "v2"))
                .setUpdatedAt(System.currentTimeMillis())
                .setDeletedAt(System.currentTimeMillis() - 10_000_000)));
    }

    private static Resource resource(long now) {
        return staticResource().setUpdatedAt(now);
    }

    private TtlPredicate predicate(long now, long ttlMillis, Object2LongMap<String> ttlByService) {
        return new TtlPredicate(now, ttlMillis, ttlByService, ResourceValidator.STRICT_VALIDATOR);
    }
}
