package ru.yandex.solomon.name.resolver.ttl;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.name.resolver.IssueTrackerNoop;
import ru.yandex.solomon.name.resolver.NameResolverLocalShards;
import ru.yandex.solomon.name.resolver.NameResolverShardFactoryStub;
import ru.yandex.solomon.name.resolver.ttl.TtlScheduler.Status;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class TtlSchedulerTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .build();

    private NameResolverShardFactoryStub shardFactory;
    private NameResolverLocalShards shards;
    private TtlScheduler scheduler;

    @Before
    public void setUp() throws Exception {
        shards = new NameResolverLocalShards();
        shardFactory = new NameResolverShardFactoryStub();
        var opts = TtlSchedulerOpts.newBuilder()
                .resourceTtl(1, TimeUnit.DAYS)
                .minUptimeToRun(10, TimeUnit.MINUTES)
                .build();
        scheduler = new TtlScheduler(opts, shards, new IssueTrackerNoop(), resource -> false, new MetricRegistry(), ForkJoinPool.commonPool(), shardFactory.clock, shardFactory.timer);
        scheduler.onUpdateServiceProviders(Map.of());
    }

    @After
    public void tearDown() {
        scheduler.close();
        shards.gracefulShutdown().join();
        shardFactory.close();
    }

    @Test
    public void noShards() throws InterruptedException {
        scheduler.actSync().await();
    }

    @Test
    public void shardUptimeTooSmall() throws InterruptedException {
        var shard = shardFactory.create("test", new AssignmentSeqNo(1, 22));
        shards.addShard(shard);
        shard.start().join();

        var resource = staticResource().setCloudId("test")
                .setDeletedAt(shardFactory.clock.millis() - TimeUnit.DAYS.toMillis(100));
        shard.update(resource).join();

        for (int index = 0; index < 10; index++) {
            scheduler.actSync().await();
        }

        assertEquals(List.of(resource), shardFactory.resourceDao.findResources("test").join());
    }

    @Test
    public void runDeleteTask() throws InterruptedException {
        var shard = shardFactory.create("test", new AssignmentSeqNo(1, 22));
        shards.addShard(shard);
        shard.start().join();

        var resource = staticResource().setCloudId("test")
                .setDeletedAt(shardFactory.clock.millis() - TimeUnit.DAYS.toMillis(100));
        shard.update(resource).join();

        var future = awaitScheduled("test").nextFuture();
        scheduler.actSync().await();
        future.join();

        assertEquals(List.of(), shardFactory.resourceDao.findResources("test").join());
    }

    private Status awaitScheduled(String shardId) throws InterruptedException {
        while (true) {
            var status = scheduler.getTtlStatus("test");
            if (status != null) {
                shardFactory.clock.passedTime(status.nextPeriod(), TimeUnit.MILLISECONDS);
                return status;
            }

            shardFactory.clock.passedTime(1, TimeUnit.HOURS);
            scheduler.actSync().await();
        }
    }
}
