package ru.yandex.solomon.name.resolver.ttl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.IssueTrackerNoop;
import ru.yandex.solomon.name.resolver.NameResolverShard;
import ru.yandex.solomon.name.resolver.NameResolverShardFactoryStub;
import ru.yandex.solomon.name.resolver.client.Resource;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class TtlTaskTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(10, TimeUnit.SECONDS)
            .build();

    private NameResolverShardFactoryStub stub;
    private NameResolverShard shard;

    @Before
    public void setUp() {
        stub = new NameResolverShardFactoryStub();
        createShard();
    }

    @Test
    public void removeAll() {
        shard.update(Arrays.asList(resource().setResourceId("alice"), resource().setResourceId("bob")), false, "").join();

        TtlTask task = new TtlTask(shard, new IssueTrackerNoop(), resource -> true);
        task.run().join();

        assertEquals(List.of(), stub.resourceDao.findResources(shard.cloudId).join());
        assertEquals(List.of(), shard.search(Selectors.of()).collect(Collectors.toList()));
    }

    @Test
    public void removeManyNotTouchParticular() {
        ArrayList<Resource> resources = new ArrayList<>();
        IntStream.range(0, 10_002)
                .mapToObj(idx -> resource().setResourceId("id-" + idx))
                .forEach(resources::add);

        var alice = resource().setResourceId("alice");
        resources.add(alice);
        Collections.shuffle(resources);
        shard.update(resources, false, "").join();

        TtlTask task = new TtlTask(shard, new IssueTrackerNoop(), resource -> !"alice".equals(resource.resourceId));
        task.run().join();

        assertEquals(List.of(alice), stub.resourceDao.findResources(shard.cloudId).join());
        assertEquals(List.of(alice), shard.search(Selectors.of()).collect(Collectors.toList()));
    }

    @Test
    public void removeNotUpdated() {
        long now = System.currentTimeMillis();
        long deadline = now - TimeUnit.HOURS.toMillis(2);
        shard.update(Arrays.asList(resource().setResourceId("alice"),
                resource().setResourceId("bob").setUpdatedAt(deadline - 1),
                resource().setResourceId("bobAnother").setService("service2").setUpdatedAt(deadline - 1),
                resource().setResourceId("bobAlive").setUpdatedAt(deadline + 10_000)), false, "").join();
        shard.update(Arrays.asList(resource().setResourceId("alice")), true, "service").join();

        TtlTask task = new TtlTask(shard, new IssueTrackerNoop(), resource -> resource.deletedAt != 0);
        task.run().join();

        assertEquals(List.of(resource().setResourceId("bobAlive").setUpdatedAt(deadline + 10_000),
                resource().setResourceId("bobAnother").setService("service2").setUpdatedAt(deadline - 1),
                resource().setResourceId("alice")), stub.resourceDao.findResources(shard.cloudId).join());
        assertEquals(List.of(resource().setResourceId("bobAlive").setUpdatedAt(deadline + 10_000),
                resource().setResourceId("bobAnother").setService("service2").setUpdatedAt(deadline - 1),
                resource().setResourceId("alice").setReplaced(true)), shard.search(Selectors.of()).collect(Collectors.toList()));
    }

    private void createShard() {
        if (shard != null) {
            shard.close();
        }

        var seqNo = new AssignmentSeqNo(1, 2);
        shard = stub.create("myCloudId", seqNo);
        shard.start().join();
    }

    private Resource resource() {
        return staticResource().setCloudId(shard.cloudId);
    }

}
