package ru.yandex.solomon.project.manager.abc.db.ydb;

import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TxControl;

import ru.yandex.solomon.project.manager.abc.AbcToProjectMapping;
import ru.yandex.solomon.project.manager.abc.db.AbcToProjectMappingDao;

import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingTable.ABC_SLUG;
import static ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingTable.PARAM_ABC_SLUG;
import static ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingTable.PARAM_PROJECT_ID;
import static ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingTable.PARAM_UPDATED_AT;
import static ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingTable.PARAM_UPDATED_BY;
import static ru.yandex.solomon.project.manager.abc.db.ydb.YdbAbcToProjectMappingTable.PROJECT_ID;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class YdbAbcToProjectMappingDao implements AbcToProjectMappingDao {

    private final SessionRetryContext retryCtx;
    private final SchemeClient schemeClient;
    private final YdbAbcToProjectMappingQuery query;

    public YdbAbcToProjectMappingDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.schemeClient = schemeClient;
        this.query = new YdbAbcToProjectMappingQuery(root);
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(query.root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(unused -> createTable(query.table, YdbAbcToProjectMappingTable::createTable));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        Function<Session, CompletableFuture<Status>> doDrop = (session) -> session.dropTable(query.table);
        return retryCtx.supplyStatus(doDrop)
                .thenAccept(status -> status.expect("cannot drop " + query.table + " table"));
    }

    @Override
    public CompletableFuture<Optional<AbcToProjectMapping>> getMapping(String abcSlug) {
        Params params = Params.of(PARAM_ABC_SLUG, utf8(abcSlug));
        return execute(query.selectOne, params)
                .thenApply(result -> {
                    var dataQueryResult = result.expect("unable select abcToProjectMapping by id");
                    var rs = dataQueryResult.getResultSet(0);
                    if (!rs.next()) {
                        return Optional.empty();
                    }
                    return Optional.of(map(rs));
                });
    }

    @Override
    public CompletableFuture<Void> setDefaultProject(AbcToProjectMapping mapping, String updatedBy) {
        var params = Params.create()
                .put(PARAM_ABC_SLUG, utf8(mapping.abcSlug()))
                .put(PARAM_UPDATED_BY, utf8(updatedBy))
                .put(PARAM_UPDATED_AT, timestamp(Instant.now()))
                .put(PARAM_PROJECT_ID, utf8(mapping.projectId()));

        return execute(query.upsert, params)
                .thenApply(result -> {
                    if (result.isSuccess()) {
                        return null;
                    }
                    result.expect("unable upsert abcToProjectMapping");
                    return null;
                });
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> createTable(String tablePath, BiFunction<String, Session, CompletableFuture<Status>> fn) {
        return schemeClient.describePath(tablePath)
                .thenCompose(exist -> {
                    if (exist.isSuccess()) {
                        return completedFuture(com.yandex.ydb.core.Status.SUCCESS);
                    }

                    return retryCtx.supplyStatus(session -> fn.apply(tablePath, session));
                })
                .thenAccept(status -> status.expect("cannot create abcToProjectMapping table " + tablePath));
    }

    public static AbcToProjectMapping map(ResultSetReader rs) {
        return new AbcToProjectMapping(
                rs.getColumn(ABC_SLUG).getUtf8(),
                rs.getColumn(PROJECT_ID).getUtf8()
        );
    }
}
