package ru.yandex.solomon.project.manager.api;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.annotation.JsonInclude;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.http.HttpMethod;
import org.springframework.http.codec.ServerCodecConfigurer;
import org.springframework.http.codec.json.Jackson2JsonDecoder;
import org.springframework.http.codec.json.Jackson2JsonEncoder;
import org.springframework.stereotype.Component;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.reactive.CorsWebFilter;
import org.springframework.web.reactive.config.ResourceHandlerRegistry;
import org.springframework.web.reactive.config.WebFluxConfigurer;
import org.springframework.web.reactive.result.method.annotation.ArgumentResolverConfigurer;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.Contact;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.SpringfoxWebConfiguration;
import springfox.documentation.spring.web.SpringfoxWebFluxConfiguration;
import springfox.documentation.spring.web.json.JacksonModuleRegistrar;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger.configuration.SwaggerCommonConfiguration;
import springfox.documentation.swagger2.configuration.Swagger2WebFluxConfiguration;
import springfox.documentation.swagger2.mappers.CompatibilityModelMapperImpl;
import springfox.documentation.swagger2.mappers.LicenseMapperImpl;
import springfox.documentation.swagger2.mappers.ModelSpecificationMapperImpl;
import springfox.documentation.swagger2.mappers.ParameterMapperImpl;
import springfox.documentation.swagger2.mappers.PropertyMapperImpl;
import springfox.documentation.swagger2.mappers.SecurityMapperImpl;
import springfox.documentation.swagger2.mappers.ServiceModelToSwagger2MapperImpl;
import springfox.documentation.swagger2.mappers.VendorExtensionsMapperImpl;
import springfox.documentation.swagger2.web.Swagger2ControllerWebFlux;

import ru.yandex.monlib.metrics.webflux.filters.HttpStatsFilter;
import ru.yandex.solomon.auth.http.AuthMethodArgumentResolver;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.http.RequireAuth;
import ru.yandex.solomon.config.DataSizeConverter;
import ru.yandex.solomon.config.protobuf.project.manager.ProjectManagerConfig;
import ru.yandex.solomon.exception.handlers.HttpApiExceptionHandler;
import ru.yandex.solomon.util.jackson.SalmonJackson;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Alexey Trushkin
 */
@Configuration
@ComponentScan(basePackageClasses = ApiContext.class)
@Import({
        ApiContext.WebFluxConfig.class,
        HttpApiExceptionHandler.class,
        SwaggerCommonConfiguration.class,
        SpringfoxWebConfiguration.class,
        SpringfoxWebFluxConfiguration.class,
        Swagger2WebFluxConfiguration.class,
        Swagger2ControllerWebFlux.class,
        ServiceModelToSwagger2MapperImpl.class,
        CompatibilityModelMapperImpl.class,
        SecurityMapperImpl.class,
        LicenseMapperImpl.class,
        VendorExtensionsMapperImpl.class,
        ParameterMapperImpl.class,
        PropertyMapperImpl.class,
        ModelSpecificationMapperImpl.class,
        HttpStatsFilter.class,
})
public class ApiContext {

    @Bean
    public JacksonModuleRegistrar excludeNullMapper() {
        return objectMapper -> objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    @Bean(name = "whiteListAuthorizer")
    public WhiteListAuthorizer whiteListAuthorizer(ProjectManagerConfig config) {
        Set<String> whiteList;
        if (config.hasWhiteListForBySubjectApiConfig()) {
            whiteList = new HashSet<>(config.getWhiteListForBySubjectApiConfig().getAllowList());
        } else {
            whiteList = Set.of();
        }
        return new WhiteListAuthorizer(whiteList);
    }

    @Bean(name = "whiteListLegacyAuthorizer")
    public WhiteListAuthorizer whiteListLegacyAuthorizer(ProjectManagerConfig config) {
        Set<String> whiteList;
        if (config.hasWhiteListForLegacyApiConfig()) {
            whiteList = new HashSet<>(config.getWhiteListForLegacyApiConfig().getAllowList());
        } else {
            whiteList = Set.of();
        }
        return new WhiteListAuthorizer(whiteList);
    }

    @Bean
    public Docket apiV3() {
        return createDocket("/api/v3", "Solomon API v3.0", "2.0", "/api/v3.*", true);
    }

    private static Docket createDocket(String groupName, String title, String version, String pathRegex, boolean enabled) {
        ApiInfo apiInfo = new ApiInfoBuilder()
                .title(title)
                .version(version)
                .contact(new Contact("Solomon dev team", "https://wiki.yandex-team.ru/solomon", "solomon@yandex-team.ru"))
                .build();

        Docket docket = new Docket(DocumentationType.SWAGGER_2)
                .groupName(groupName)
                .useDefaultResponseMessages(false)
                .genericModelSubstitutes(CompletableFuture.class)
                .apiInfo(apiInfo)
                .enable(enabled)
                .ignoredParameterTypes(RequireAuth.class, PageOptions.class);

        return docket.select()
                .paths(PathSelectors.regex(pathRegex))
                .build();
    }

    @Bean
    public CorsWebFilter corsFilter() {
        var corsConfig = new CorsConfiguration();
        corsConfig.setAllowedOriginPatterns(List.of("*"));
        corsConfig.addAllowedMethod(HttpMethod.GET);
        corsConfig.addAllowedMethod(HttpMethod.POST);
        corsConfig.addAllowedMethod(HttpMethod.PUT);
        corsConfig.addAllowedMethod(HttpMethod.DELETE);
        corsConfig.addAllowedMethod(HttpMethod.PATCH);
        corsConfig.addAllowedMethod(HttpMethod.OPTIONS);
        corsConfig.addAllowedHeader("cookie");
        corsConfig.addAllowedHeader("content-type");
        corsConfig.addAllowedHeader("x-yandex-login");
        corsConfig.setAllowCredentials(Boolean.TRUE);
        corsConfig.setMaxAge(3600L);

        return new CorsWebFilter(exchange -> {
            String servletPath = exchange.getRequest().getPath().toString();
            if (servletPath.startsWith("/api") || servletPath.startsWith("/rest") || servletPath.startsWith("/monitoring/v1")) {
                return corsConfig;
            }
            return null;
        });
    }

    @Component
    public static class WebFluxConfig implements WebFluxConfigurer {
        @Autowired
        private HttpAuthenticator authenticator;
        @Autowired
        private ProjectManagerConfig config;

        @Override
        public void configureHttpMessageCodecs(ServerCodecConfigurer configurer) {
            var codecs = configurer.defaultCodecs();

            var httpServerConfig = config.getHttpServerConfig();
            if (httpServerConfig.hasMaxRequestSize()) {
                codecs.maxInMemorySize(DataSizeConverter.toBytesInt(httpServerConfig.getMaxRequestSize()));
            }

            codecs.jackson2JsonEncoder(new Jackson2JsonEncoder(SalmonJackson.getObjectMapper()));
            codecs.jackson2JsonDecoder(new Jackson2JsonDecoder(SalmonJackson.getObjectMapper()));
        }

        @Override
        public void configureArgumentResolvers(ArgumentResolverConfigurer configurer) {
            configurer.addCustomResolver(new AuthMethodArgumentResolver(authenticator));
        }

        @Override
        public void addResourceHandlers(ResourceHandlerRegistry registry) {
            registry.addResourceHandler("/swagger-ui/**")
                    .addResourceLocations("classpath:/META-INF/resources/webjars/springfox-swagger-ui/")
                    .resourceChain(false);
        }
    }
}
