package ru.yandex.solomon.project.manager.api.v3.intranet.dto;

import java.time.Instant;
import java.util.stream.Collectors;

import com.google.protobuf.util.Timestamps;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsResponse;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.solomon.config.protobuf.project.manager.ProjectsConfig;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.ydb.page.PagedResult;

/**
 * @author Alexey Trushkin
 */
public class ProjectDtoConverter {

    public Project fromEntity(ru.yandex.solomon.core.db.model.Project project) {
        return Project.newBuilder()
                .setProjectId(project.getId())
                .setName(project.getName())
                .setDescription(project.getDescription())
                .setCreatedAt(Timestamps.fromMillis(project.getCreatedAtMillis()))
                .setModifiedAt(Timestamps.fromMillis(project.getUpdatedAtMillis()))
                .setCreatedBy(project.getCreatedBy())
                .setModifiedBy(project.getUpdatedBy())
                .setAbcService(project.getAbcService())
                .putAllLabels(project.getLabels())
                .setEtag(String.valueOf(project.getVersion()))
                .setOwner(project.getOwner())
                .build();
    }

    public ListProjectsResponse fromEntity(PagedResult<ru.yandex.solomon.core.db.model.Project> result) {
        String token;
        if (result.getCurrentPage() + 1 >= result.getPagesCount()) {
            token = "";
        } else {
            token = fromEntity(result.getPageSize(), result.getCurrentPage() + 1);
        }
        return ListProjectsResponse.newBuilder()
                .addAllProjects(result.getResult().stream()
                        .map(this::fromEntity)
                        .collect(Collectors.toList()))
                .setNextPageToken(token)
                .setTotalCount(result.getTotalCount())
                .build();
    }

    private String fromEntity(int pageSize, int currentPage) {
        return new YdbProjectsDao.PageToken(pageSize, currentPage).encode();
    }

    public ru.yandex.solomon.core.db.model.Project toEntity(CreateProjectRequest request, String subjectId, ProjectsConfig projectsConfig) {
        Instant now = Instant.now();
        return ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId(request.getProjectId())
                .setName(request.getName())
                .setDescription(request.getDescription())
                .setLabels(request.getLabelsMap())
                .setAbcService(request.getAbcService())
                .setOwner(subjectId)
                .setCreatedBy(subjectId)
                .setCreatedAt(now)
                .setOnlyAuthPush(true)
                .setOnlyAuthRead(false)
                .setOnlyMetricNameShards(projectsConfig.getOnlyMetricNameShardsInNewProjects())
                .setOnlyNewFormatWrites(false)
                .setOnlyNewFormatReads(false)
                .build();
    }

    public ru.yandex.solomon.core.db.model.Project toEntity(
            UpdateProjectRequest request,
            String subjectId,
            ru.yandex.solomon.core.db.model.Project prev)
    {
        Instant now = Instant.now();
        return ru.yandex.solomon.core.db.model.Project.newBuilder()
                .setId(request.getProjectId())
                .setName(request.getName())
                .setDescription(request.getDescription())
                .setLabels(request.getLabelsMap())
                .setAbcService(request.getAbcService())
                .setUpdatedBy(subjectId)
                .setUpdatedAt(now)
                .setAbcService(request.getAbcService())
                .setOwner(prev.getOwner())
                .setCreatedBy(prev.getCreatedBy())
                .setCreatedAt(prev.getCreatedAt())
                .setOnlyAuthPush(prev.isOnlyAuthPush())
                .setOnlyAuthRead(prev.isOnlyAuthRead())
                .setOnlyMetricNameShards(prev.isOnlyMetricNameShards())
                .setOnlyNewFormatWrites(prev.isOnlyNewFormatWrites())
                .setOnlyNewFormatReads(prev.isOnlyNewFormatReads())
                .setMetricNameLabel(prev.getMetricNameLabel())
                .setAcl(prev.getAcl())
                .setVersion(parseVersion(request.getEtag()))
                .build();
    }

    public int parseVersion(String etag) {
        int version = -1;
        try {
            if (!StringUtils.isEmpty(etag)) {
                version = Integer.parseInt(etag);
            }
        } catch (NumberFormatException ignore) {
            throw new BadRequestException("Etag wrong format");
        }
        return version;
    }


}
