package ru.yandex.solomon.project.manager.api.v3.intranet.grpc;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.project.manager.v3.CreateProjectRequest;
import ru.yandex.project.manager.v3.CreateProjectResponse;
import ru.yandex.project.manager.v3.DeleteOneRequest;
import ru.yandex.project.manager.v3.DeleteOneResponse;
import ru.yandex.project.manager.v3.ExistsRequest;
import ru.yandex.project.manager.v3.ExistsResponse;
import ru.yandex.project.manager.v3.FindByIdRequest;
import ru.yandex.project.manager.v3.FindByIdResponse;
import ru.yandex.project.manager.v3.FindInProjectsRequest;
import ru.yandex.project.manager.v3.FindRequest;
import ru.yandex.project.manager.v3.FindResponse;
import ru.yandex.project.manager.v3.FindV3Request;
import ru.yandex.project.manager.v3.FindV3Response;
import ru.yandex.project.manager.v3.GetAllRequest;
import ru.yandex.project.manager.v3.GetAllResponse;
import ru.yandex.project.manager.v3.PartialUpdateRequest;
import ru.yandex.project.manager.v3.PartialUpdateResponse;
import ru.yandex.project.manager.v3.ProjectLegacyServiceGrpc;
import ru.yandex.project.manager.v3.UpsertProjectsRequest;
import ru.yandex.project.manager.v3.UpsertProjectsResponse;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.client.ProjectDtoConverter;
import ru.yandex.solomon.core.db.model.ProjectPermission;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.grpc.handler.GrpcMethodHandler;
import ru.yandex.solomon.project.manager.api.WhiteListAuthorizer;
import ru.yandex.solomon.project.manager.api.v3.intranet.ProjectChangeListener;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Alexey Trushkin
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcProjectLegacyService extends ProjectLegacyServiceGrpc.ProjectLegacyServiceImplBase implements GrpcService {

    private static final Logger logger = LoggerFactory.getLogger(GrpcProjectLegacyService.class);

    private final WhiteListAuthorizer whiteListAuthorizer;
    private final ProjectsDao projectsDao;
    private final FeatureFlagsHolder featureFlagsHolder;
    private final List<ProjectChangeListener> projectChangeListeners;
    private final MetricRegistry metricRegistry;

    public GrpcProjectLegacyService(
            @Qualifier("whiteListLegacyAuthorizer") WhiteListAuthorizer whiteListAuthorizer,
            ProjectsDao projectsDao,
            FeatureFlagsHolder featureFlagsHolder,
            List<ProjectChangeListener> projectChangeListeners,
            MetricRegistry metricRegistry)
    {
        this.whiteListAuthorizer = whiteListAuthorizer;
        this.projectsDao = projectsDao;
        this.featureFlagsHolder = featureFlagsHolder;
        this.projectChangeListeners = projectChangeListeners;
        this.metricRegistry = metricRegistry;
    }

    @Override
    public void insert(CreateProjectRequest request, StreamObserver<CreateProjectResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doInsert, request, responseObserver);
    }

    private CompletableFuture<CreateProjectResponse> doInsert(CreateProjectRequest createProjectRequest, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        var projectEntity = ProjectDtoConverter.toModel(createProjectRequest.getProject()).get();
        return projectsDao.insert(projectEntity)
                .thenCompose(aBoolean -> {
                    List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                    for (var listener : projectChangeListeners) {
                        futures.add(listener.create(projectEntity)
                                .handle((unused, throwable) -> handleSafe(throwable, listener, "createLegacy")));
                    }
                    return CompletableFutures.allOfVoid(futures)
                            .thenApply(unused -> aBoolean);
                })
                .thenApply(aBoolean -> CreateProjectResponse.newBuilder().setResult(aBoolean).build());
    }

    @Override
    public void findById(FindByIdRequest request, StreamObserver<FindByIdResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doFindById, request, responseObserver);
    }

    private CompletableFuture<FindByIdResponse> doFindById(FindByIdRequest findByIdRequest, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        return projectsDao.findById(findByIdRequest.getId())
                .thenApply(ProjectDtoConverter::toProto);
    }

    @Override
    public void getAll(GetAllRequest request, StreamObserver<GetAllResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doGetAll, request, responseObserver);
    }

    private CompletableFuture<GetAllResponse> doGetAll(GetAllRequest getAllRequest, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        return projectsDao.findAllNames()
                .thenApply(ProjectDtoConverter::toProto);
    }

    @Override
    public void find(FindRequest request, StreamObserver<FindResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doFind, request, responseObserver);
    }

    private CompletableFuture<FindResponse> doFind(FindRequest request, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        var set = request.getFilterByPermissionsList().stream()
                .map(ProjectPermission::valueOf)
                .collect(Collectors.toSet());
        var permissions = set.isEmpty() ? EnumSet.noneOf(ProjectPermission.class) : EnumSet.copyOf(set);
        var page = new PageOptions(request.getSize(), request.getCurrent());
        return projectsDao.find(request.getText(), request.getAbcFilter(), request.getLogin(), permissions, page)
                .thenApply(ProjectDtoConverter::toProto);
    }

    @Override
    public void findInProjects(FindInProjectsRequest request, StreamObserver<FindResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doFindInProjects, request, responseObserver);
    }

    private CompletableFuture<FindResponse> doFindInProjects(FindInProjectsRequest request, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        var page = new PageOptions(request.getSize(), request.getCurrent());
        return projectsDao.findInProjects(request.getText(), request.getAbcFilter(), new HashSet<>(request.getProjectIdsList()), page)
                .thenApply(ProjectDtoConverter::toProto);
    }

    @Override
    public void findV3(FindV3Request request, StreamObserver<FindV3Response> responseObserver) {
        GrpcMethodHandler.handle(this::doFindV3, request, responseObserver);
    }

    private CompletableFuture<FindV3Response> doFindV3(FindV3Request request, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        return projectsDao.findV3(request.getText(), request.getPageSize(), request.getPageToken())
                .thenApply(ProjectDtoConverter::toProto);
    }

    @Override
    public void partialUpdate(PartialUpdateRequest request, StreamObserver<PartialUpdateResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doPartialUpdate, request, responseObserver);
    }

    private CompletableFuture<PartialUpdateResponse> doPartialUpdate(PartialUpdateRequest request, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        var projectEntity = ProjectDtoConverter.toModel(request.getProject()).get();
        return projectsDao.partialUpdate(projectEntity, request.getCanChangeOwner(), request.getCanChangeInternalOptions(), request.getCanUpdateOldFields())
                .thenCompose(projectOptional -> {
                    List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                    for (var listener : projectChangeListeners) {
                        futures.add(listener.update(projectEntity)
                                .handle((unused, throwable) -> handleSafe(throwable, listener, "updateLegacy")));
                    }
                    return CompletableFutures.allOfVoid(futures)
                            .thenApply(unused -> projectOptional);
                })
                .thenApply(ProjectDtoConverter::toProtoUpdate);
    }

    @Override
    public void upsertProjects(UpsertProjectsRequest request, StreamObserver<UpsertProjectsResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doUpsertProjects, request, responseObserver);
    }

    private CompletableFuture<UpsertProjectsResponse> doUpsertProjects(UpsertProjectsRequest request, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        return projectsDao.upsertProjects(ProjectDtoConverter.toModel(request))
                .thenApply(unused -> UpsertProjectsResponse.getDefaultInstance());
    }

    @Override
    public void deleteOne(DeleteOneRequest request, StreamObserver<DeleteOneResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doDeleteOne, request, responseObserver);
    }

    private CompletableFuture<DeleteOneResponse> doDeleteOne(DeleteOneRequest request, AuthSubject authSubject) {
        if (!SolomonTeam.isMember(authSubject)) {
            whiteListAuthorizer.authorize(authSubject);
            if (!request.getSkipValidation()) {
                if (featureFlagsHolder.hasFlag(FeatureFlag.USE_PM_JUGGLER_INTEGRATION, FeatureFlag.USE_PM_JUGGLER_INTEGRATION.name())) {
                    return CompletableFuture.failedFuture(new AuthorizationException("Can't delete project '" + request.getId() + "' use ticket for that."));
                }
            }
        }
        return projectsDao.deleteOne(request.getId(), request.getSkipValidation())
                .thenCompose(aBoolean -> {
                    List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                    for (var listener : projectChangeListeners) {
                        futures.add(listener.delete(DeleteProjectRequest.newBuilder()
                                .setProjectId(request.getId())
                                .build())
                                .handle((unused, throwable) -> handleSafe(throwable, listener, "deleteLegacy")));
                    }
                    return CompletableFutures.allOfVoid(futures)
                            .thenApply(unused -> aBoolean);
                })
                .thenApply(aBoolean -> DeleteOneResponse.newBuilder().setResult(aBoolean).build());
    }

    @Override
    public void exists(ExistsRequest request, StreamObserver<ExistsResponse> responseObserver) {
        GrpcMethodHandler.handle(this::doExists, request, responseObserver);
    }

    private CompletableFuture<ExistsResponse> doExists(ExistsRequest request, AuthSubject authSubject) {
        whiteListAuthorizer.authorize(authSubject);
        return projectsDao.exists(request.getId())
                .thenApply(aBoolean -> ExistsResponse.newBuilder().setResult(aBoolean).build());
    }


    private Void handleSafe(@Nullable Throwable throwable, ProjectChangeListener listener, String method) {
        Labels labels = Labels.of("listener", listener.getClass().getSimpleName(), "method", method);
        if (throwable != null) {
            logger.error("Can't handle '{}' for listener type {}", method, listener.getClass().getSimpleName());
            metricRegistry.rate("project.changes_push.completed_error", labels).inc();
        } else {
            metricRegistry.rate("project.changes_push.completed_ok", labels).inc();
        }
        return null;
    }
}
