package ru.yandex.solomon.project.manager.api.v3.intranet.grpc;

import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.stub.StreamObserver;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.GrpcService;
import ru.yandex.monitoring.api.v3.Operation;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectBySubjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectBySubjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.GetProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectOperationsRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectOperationsResponse;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsResponse;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.ProjectServiceGrpc;
import ru.yandex.monitoring.api.v3.project.manager.Subject;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectBySubjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.grpc.handler.GrpcMethodHandler;
import ru.yandex.solomon.operation.OperationDtoConverter;
import ru.yandex.solomon.project.manager.api.WhiteListAuthorizer;
import ru.yandex.solomon.project.manager.api.v3.intranet.ProjectService;

/**
 * @author Alexey Trushkin
 */
@Component
@ParametersAreNonnullByDefault
public class GrpcProjectService extends ProjectServiceGrpc.ProjectServiceImplBase implements GrpcService {

    private final ProjectService service;
    private final WhiteListAuthorizer whiteListAuthorizer;
    private final OperationDtoConverter operationConverter;

    @Autowired
    public GrpcProjectService(
            ProjectService service,
            @Qualifier("whiteListAuthorizer") WhiteListAuthorizer whiteListAuthorizer)
    {
        this.service = service;
        this.whiteListAuthorizer = whiteListAuthorizer;
        operationConverter = new OperationDtoConverter();
    }

    @Override
    public void get(GetProjectRequest request, StreamObserver<Project> responseObserver) {
        GrpcMethodHandler.handle(service::get, request, responseObserver);
    }

    @Override
    public void list(ListProjectsRequest request, StreamObserver<ListProjectsResponse> responseObserver) {
        GrpcMethodHandler.handle(service::list, request, responseObserver);
    }

    @Override
    public void create(CreateProjectRequest request, StreamObserver<Operation> responseObserver) {
        GrpcMethodHandler.handle(service::create, request, responseObserver, operationConverter::wrap);
    }

    @Override
    public void update(UpdateProjectRequest request, StreamObserver<Operation> responseObserver) {
        GrpcMethodHandler.handle(service::update, request, responseObserver, operationConverter::wrap);
    }

    @Override
    public void delete(DeleteProjectRequest request, StreamObserver<Operation> responseObserver) {
        GrpcMethodHandler.handle(service::delete, request, responseObserver, operationConverter::wrap);
    }

    @Override
    public void createBySubject(CreateProjectBySubjectRequest request, StreamObserver<Operation> responseObserver) {
        GrpcMethodHandler.handle(service::create, request.getCreateProjectRequest(), responseObserver, operationConverter::wrap,
                authorizeWhiteList(request.getSubject()));
    }

    @Override
    public void updateBySubject(UpdateProjectBySubjectRequest request, StreamObserver<Operation> responseObserver) {
        GrpcMethodHandler.handle(service::update, request.getUpdateProjectRequest(), responseObserver, operationConverter::wrap,
                authorizeWhiteList(request.getSubject()));
    }

    @Override
    public void deleteBySubject(DeleteProjectBySubjectRequest request, StreamObserver<Operation> responseObserver) {
        GrpcMethodHandler.handle(service::delete, request.getDeleteProjectRequest(), responseObserver, operationConverter::wrap,
                authorizeWhiteList(request.getSubject()));
    }

    private Supplier<AuthSubject> authorizeWhiteList(Subject subject) {
        return whiteListAuthorizer.authorizeSupplier(subject);
    }

    @Override
    public void listOperations(ListProjectOperationsRequest request, StreamObserver<ListProjectOperationsResponse> responseObserver) {
        GrpcMethodHandler.SimpleHandlerFunc<ListProjectOperationsRequest, ListProjectOperationsResponse> returnEmpty =
                (request1, subject) -> ListProjectOperationsResponse.getDefaultInstance();
        GrpcMethodHandler.handle(returnEmpty, request, responseObserver);
    }
}
