package ru.yandex.solomon.project.manager.api.v3.intranet.impl;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.project.manager.GetAbcDefaultProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.SetAbcDefaultProjectRequest;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.project.manager.abc.AbcToProjectMapping;
import ru.yandex.solomon.project.manager.abc.db.AbcToProjectMappingDao;
import ru.yandex.solomon.project.manager.api.v3.intranet.AbcToProjectMappingService;
import ru.yandex.solomon.project.manager.api.v3.intranet.dto.ProjectDtoConverter;
import ru.yandex.solomon.project.manager.api.v3.intranet.validators.AbcToProjectMappingValidator;
import ru.yandex.solomon.ydb.page.PageOptions;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class AbcToProjectMappingServiceImpl implements AbcToProjectMappingService {

    private static final Logger logger = LoggerFactory.getLogger(AbcToProjectMappingServiceImpl.class);
    private final AbcToProjectMappingDao abcToProjectMappingDao;
    private final ProjectsDao projectsDao;
    private final ProjectDtoConverter projectDtoConverter;
    private final AbcToProjectMappingValidator validator;
    private final Authorizer authorizer;

    public AbcToProjectMappingServiceImpl(AbcToProjectMappingDao abcToProjectMappingDao, ProjectsDao projectsDao, Authorizer authorizer) {
        this.abcToProjectMappingDao = abcToProjectMappingDao;
        this.projectsDao = projectsDao;
        this.authorizer = authorizer;
        projectDtoConverter = new ProjectDtoConverter();
        validator = new AbcToProjectMappingValidator(projectsDao);
    }

    @Override
    public CompletableFuture<Project> getDefaultProjectForAbc(GetAbcDefaultProjectRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> {
            var abcSlug = request.getAbcService();
            validator.validate(abcSlug);
            return abcToProjectMappingDao.getMapping(abcSlug)
                    .thenCompose(mapping -> {
                        if (mapping.isEmpty()) {
                            return projectsDao.find("", abcSlug, "", null, new PageOptions(2, 0))
                                    .thenApply(result -> {
                                        if (result.getResult().size() != 1) {
                                            logger.info("For abc '{}' found '{}' projects", abcSlug, result.getTotalCount());
                                            throw new NotFoundException(String.format("no single project for abc '%s'", abcSlug));
                                        }
                                        return result.getResult().get(0);
                                    });
                        } else {
                            return projectsDao.findById(mapping.get().projectId())
                                    .thenApply(project -> project.orElseThrow(() -> ProjectServiceImpl.projectNotFound(mapping.get().projectId())));
                        }
                    })
                    .thenApply(projectDtoConverter::fromEntity);
        });
    }

    @Override
    public CompletableFuture<Void> setDefaultProjectForAbc(SetAbcDefaultProjectRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> validator.validate(request.getAbcService(), request.getProjectId())
                .thenCompose(unused -> authorizer.authorize(subject, AuthorizationObject.abc(request.getAbcService(), "administration"), Permission.CONFIGS_UPDATE))
                .thenCompose(unused -> abcToProjectMappingDao.setDefaultProject(new AbcToProjectMapping(request.getAbcService(), request.getProjectId()), subject.getUniqueId())));
    }
}
