package ru.yandex.solomon.project.manager.api.v3.intranet.impl;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Empty;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.project.manager.CreateProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.GetProjectRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsRequest;
import ru.yandex.monitoring.api.v3.project.manager.ListProjectsResponse;
import ru.yandex.monitoring.api.v3.project.manager.Project;
import ru.yandex.monitoring.api.v3.project.manager.UpdateProjectRequest;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.abc.validator.AbcServiceFieldValidator;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.AuthorizationObject;
import ru.yandex.solomon.auth.Authorizer;
import ru.yandex.solomon.auth.SolomonTeam;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.protobuf.project.manager.ProjectsConfig;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.core.exceptions.MethodNotAllowedException;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.core.exceptions.NotOwnerException;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.project.manager.api.v3.intranet.ProjectChangeListener;
import ru.yandex.solomon.project.manager.api.v3.intranet.ProjectService;
import ru.yandex.solomon.project.manager.api.v3.intranet.dto.ProjectDtoConverter;
import ru.yandex.solomon.project.manager.api.v3.intranet.validators.ProjectValidator;

/**
 * @author Alexey Trushkin
 */
@Component
@ParametersAreNonnullByDefault
public class ProjectServiceImpl implements ProjectService {

    private static final Logger logger = LoggerFactory.getLogger(ProjectServiceImpl.class);

    private final Authorizer authorizer;
    private final ProjectsDao projectsDao;
    private final List<ProjectChangeListener> projectChangeListeners;
    private final MetricRegistry metricRegistry;
    private final FeatureFlagsHolder featureFlagsHolder;
    private final ProjectsConfig projectsConfig;
    private final ProjectDtoConverter projectDtoConverter;
    private final ProjectValidator validator;
    private final AbcServiceFieldValidator abcServiceFieldValidator;
    private final Boolean forbidCreate;

    public ProjectServiceImpl(
            Authorizer authorizer,
            ProjectsDao projectsDao,
            AbcServiceFieldValidator abcServiceFieldValidator,
            List<ProjectChangeListener> projectChangeListeners,
            MetricRegistry metricRegistry,
            FeatureFlagsHolder featureFlagsHolder,
            @Qualifier("ForbidProjectCreation") Optional<Boolean> forbidCreate,
            Optional<ProjectsConfig> projectsConfig)
    {
        this.authorizer = authorizer;
        this.projectsDao = projectsDao;
        this.projectChangeListeners = projectChangeListeners;
        this.metricRegistry = metricRegistry;
        this.featureFlagsHolder = featureFlagsHolder;
        this.projectsConfig = projectsConfig.orElse(ProjectsConfig.getDefaultInstance());
        projectDtoConverter = new ProjectDtoConverter();
        validator = new ProjectValidator();
        this.abcServiceFieldValidator = abcServiceFieldValidator;
        this.forbidCreate = forbidCreate.orElse(false);
    }

    @Override
    public CompletableFuture<Project> get(GetProjectRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> {
            validator.validate(request);
            return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_GET)
                    .thenCompose(aVoid -> projectsDao.findById(request.getProjectId()))
                    .thenApply(project -> project.orElseThrow(() -> projectNotFound(request.getProjectId())))
                    .thenApply(projectDtoConverter::fromEntity);
        });
    }

    @Override
    public CompletableFuture<ListProjectsResponse> list(ListProjectsRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> {
            String login = AuthSubject.getLogin(subject)
                    .orElse(subject.getUniqueId());
            var pageToken = YdbProjectsDao.PageToken.decode(request.getPageToken(), request.getPageSize());
            if (request.getFilterByRoleList().isEmpty()) {
                return projectsDao.find(request.getFilterByName(), request.getFilterByAbcService(), login, null, pageToken.toPageOptions());
            }
            return authorizer.getAvailableAuthorizationObjects(subject, mapRoles(request.getFilterByRoleList()), EnumSet.of(AuthorizationObject.Type.CLASSIC))
                    .thenCompose(resultData -> {
                        if (resultData.objectIds().isEmpty()) {
                            return projectsDao.find(request.getFilterByName(), request.getFilterByAbcService(), login, null, pageToken.toPageOptions());
                        }
                        var projectIds = resultData.getClassicObjects().stream()
                                .map(AuthorizationObject.ClassicAuthorizationObject::projectId)
                                .collect(Collectors.toSet());
                        return projectsDao.findInProjects(request.getFilterByName(), request.getFilterByAbcService(), projectIds, pageToken.toPageOptions());
                    });
        }).thenApply(projectDtoConverter::fromEntity);
    }

    @Override
    public CompletableFuture<Project> create(CreateProjectRequest request, AuthSubject subject) {
        if (forbidCreate) {
            return CompletableFuture.failedFuture(new MethodNotAllowedException(
                    "Manual project creation is forbidden. " +
                            "Please create cloud in resource manager and wait for synchronization."));
        }
        return CompletableFutures.safeCall(() -> {
            validator.validate(request);
            var project = projectDtoConverter.toEntity(request, subject.getUniqueId(), projectsConfig);
            return abcServiceFieldValidator.validate(project.getAbcService(), true)
                    .thenCompose(unit -> projectsDao.insert(project)
                            .thenApply(inserted -> {
                                if (!inserted) {
                                    throw new ConflictException(String.format("project with id '%s' already exists", project.getId()));
                                }
                                return project;
                            })
                            .thenCompose(result -> {
                                List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                                for (var listener : projectChangeListeners) {
                                    futures.add(listener.create(result)
                                            .handle((unused, throwable) -> handleSafe(throwable, listener, "create")));
                                }
                                return CompletableFutures.allOfVoid(futures)
                                        .thenApply(unused -> result);
                            })
                            .thenApply(projectDtoConverter::fromEntity));
        });
    }

    @Override
    public CompletableFuture<Project> update(UpdateProjectRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> {
            validator.validate(request);
            return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                    .thenCompose(account -> abcServiceFieldValidator.validate(request.getAbcService(), true))
                    .thenCompose(aVoid -> projectsDao.findById(request.getProjectId()))
                    .thenApply(project -> project.orElseThrow(() -> projectNotFound(request.getProjectId())))
                    .thenApply(project -> projectDtoConverter.toEntity(request, subject.getUniqueId(), project))
                    .thenCompose(project -> projectsDao.partialUpdate(project, false, false, false)
                            .thenApply(updatedProject -> {
                                if (updatedProject.isPresent()) {
                                    return updatedProject.get();
                                }
                                throw projectIsOutOfDate(project.getId(), project.getVersion());
                            }))
                    .thenCompose(result -> {
                        List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                        for (var listener : projectChangeListeners) {
                            futures.add(listener.update(result)
                                    .handle((unused, throwable) -> handleSafe(throwable, listener, "update")));
                        }
                        return CompletableFutures.allOfVoid(futures)
                                .thenApply(unused -> result);
                    })
                    .thenApply(projectDtoConverter::fromEntity);
        });
    }

    @Override
    public CompletableFuture<Empty> delete(DeleteProjectRequest request, AuthSubject subject) {
        return CompletableFutures.safeCall(() -> {
            validator.validate(request);
            return authorizer.authorize(subject, request.getProjectId(), Permission.CONFIGS_UPDATE)
                    .thenCompose(account -> {
                        if (!account.can(Permission.PROJECTS_DELETE)) {
                            if (!SolomonTeam.isMember(subject)) {
                                if (featureFlagsHolder.hasFlag(FeatureFlag.USE_PM_JUGGLER_INTEGRATION, FeatureFlag.USE_PM_JUGGLER_INTEGRATION.name())) {
                                    return CompletableFuture.failedFuture(new AuthorizationException("Can't delete project '" + request.getProjectId() + "' use ticket for that."));
                                }
                            }
                        }
                        // for cloud accounts we check permissions in AccessService instead of
                        // relying on permissions stored in project
                        boolean canDeleteByNonOwner =
                                account.getAuthType() == AuthType.IAM ||
                                        account.can(Permission.CONFIGS_UPDATE_ANY) ||
                                        account.can(Permission.PROJECTS_DELETE);
                        return projectsDao.findById(request.getProjectId())
                                .thenAccept(projectO -> {
                                    if (projectO.isEmpty()) {
                                        throw projectNotFound(request.getProjectId());
                                    }

                                    String owner = projectO.get().getOwner();
                                    if (!canDeleteByNonOwner && !account.getId().equals(owner)) {
                                        throw new NotOwnerException("you have no permissions to delete project", owner);
                                    }
                                })
                                .thenCompose(ignore -> {
                                    List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                                    for (var listener : projectChangeListeners) {
                                        futures.add(listener.preDeleteAction(request)
                                                .handle((unused, throwable) -> handleSafe(throwable, listener, "preDeleteAction")));
                                    }
                                    return CompletableFutures.allOfVoid(futures);
                                })
                                .thenCompose(aVoid -> projectsDao.deleteOne(request.getProjectId(), SolomonTeam.isMember(subject)));
                    })
                    .thenCompose(result -> {
                        if (!result) {
                            return CompletableFuture.completedFuture(result);
                        }
                        List<CompletableFuture<Void>> futures = new ArrayList<>(projectChangeListeners.size());
                        for (var listener : projectChangeListeners) {
                            futures.add(listener.delete(request)
                                    .handle((unused, throwable) -> handleSafe(throwable, listener, "delete")));
                        }
                        return CompletableFutures.allOfVoid(futures)
                                .thenApply(unused -> result);
                    })
                    .thenApply(unused -> Empty.getDefaultInstance());
        });
    }

    public static NotFoundException projectNotFound(String id) {
        return new NotFoundException(String.format("no project with id '%s'", id));
    }

    private static ConflictException projectIsOutOfDate(String projectId, int version) {
        String message = String.format(
                "project %s with version %s is out of date",
                projectId,
                version
        );
        return new ConflictException(message);
    }

    private Set<Role> mapRoles(List<ru.yandex.monitoring.api.v3.project.manager.Role> filterByRoleList) {
        Set<Role> result = new HashSet<>();
        for (ru.yandex.monitoring.api.v3.project.manager.Role role : filterByRoleList) {
            switch (role) {
                case ROLE_UNSPECIFIED, UNRECOGNIZED -> {
                }
                case ROLE_VIEWER -> result.add(Role.VIEWER);
                case ROLE_PROJECT_ADMIN -> result.add(Role.PROJECT_ADMIN);
                case ROLE_BETA_TESTER -> result.add(Role.BETA_TESTER);
                case ROLE_MONITORING_PROJECT_DELETER -> result.add(Role.PROJECT_DELETER);
                case ROLE_MUTER -> result.add(Role.MUTER);
                case ROLE_MONITORING_ADMIN -> result.add(Role.ADMIN);
                case ROLE_EDITOR -> result.add(Role.EDITOR);
                case ROLE_PUSHER -> result.add(Role.PUSHER);
                case ROLE_SERVICE_PROVIDER_ADMIN -> result.add(Role.SERVICE_PROVIDER_ADMIN);
                case ROLE_SERVICE_PROVIDER_ALERT_EDITOR -> result.add(Role.SERVICE_PROVIDER_ALERT_EDITOR);
            }
        }
        return result;
    }

    private Void handleSafe(@Nullable Throwable throwable, ProjectChangeListener listener, String method) {
        Labels labels = Labels.of("listener", listener.getClass().getSimpleName(), "method", method);
        if (throwable != null) {
            logger.error("Can't handle '{}' for listener type {}", method, listener.getClass().getSimpleName());
            metricRegistry.rate("project.changes_push.completed_error", labels).inc();
        } else {
            metricRegistry.rate("project.changes_push.completed_ok", labels).inc();
        }
        return null;
    }
}
