package ru.yandex.solomon.project.manager.api.v3.intranet.impl.listener;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpStatusClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.ResponseStatus;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.api.v3.project.manager.DeleteProjectRequest;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.project.manager.SolomonGatewayConfig;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.project.manager.api.v3.intranet.ProjectChangeListener;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;
import ru.yandex.solomon.util.future.RetryConfig;

import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class SolomonProjectChangeListener implements ProjectChangeListener {

    private static final Logger logger = LoggerFactory.getLogger(SolomonProjectChangeListener.class);
    private static final Duration DEFAULT_REQUEST_TIMEOUT_MILLIS = Duration.ofSeconds(30);
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(throwable -> !(throwable.getCause() instanceof ClientError))
            .withNumRetries(2)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final HttpClient httpClient;

    private final ObjectMapper mapper = new ObjectMapper();
    private final HttpClientMetrics metrics;
    private final SolomonGatewayConfig config;
    private final Supplier<String> authHeaderProvider;
    private final Supplier<String> authHeaderValueProvider;

    public SolomonProjectChangeListener(
            Supplier<String> authHeaderProvider,
            Supplier<String> authHeaderValueProvider,
            MetricRegistry registry,
            SolomonGatewayConfig config,
            ExecutorService executor)
    {
        this.authHeaderProvider = authHeaderProvider;
        this.authHeaderValueProvider = authHeaderValueProvider;
        this.metrics = new HttpClientMetrics("solomon_gateway", registry);
        this.config = config;
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .executor(executor)
                .build();
    }

    @Override
    public CompletableFuture<Void> create(Project result) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> update(Project result) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> delete(DeleteProjectRequest request) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> preDeleteAction(DeleteProjectRequest request) {
        return executeWithRetries(Void.class, Map.of(), request.getProjectId(), "/deleteEntities");
    }


    private <T> CompletableFuture<T> request(Class<T> clz, Object requestDto, CompletableFuture<T> resultFuture, String projectId, String endpoint) {
        UUID uuid = UUID.randomUUID();
        String uri = config.getApiUrl() + "/api/v2/projects/" + projectId + endpoint;
        String body;
        try {
            body = mapper.writeValueAsString(requestDto);
        } catch (JsonProcessingException e) {
            return CompletableFuture.failedFuture(new RuntimeException("Can't serialize model", e));
        }
        var request = HttpRequest.newBuilder(URI.create(uri))
                .POST(HttpRequest.BodyPublishers.ofString(body))
                .header(authHeaderProvider.get(), authHeaderValueProvider.get())
                .timeout(DEFAULT_REQUEST_TIMEOUT_MILLIS)
                .build();
        logger.info("Project request:\n{}\n{}\n{}", uuid, request, body);
        var future = wrapMetrics("/api/v2/projects/:projectId" + endpoint, metrics -> httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(response -> {
                    handleException(metrics, response, uuid);
                    if (clz == Void.class) {
                        return null;
                    }
                    try {
                        return mapper.readValue(response.body(), clz);
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }));
        CompletableFutures.whenComplete(future, resultFuture);
        return future;
    }

    private <R> CompletableFuture<R> wrapMetrics(String endpoint, Function<HttpClientMetrics.Endpoint, CompletableFuture<R>> func) {
        var endpointMetrics = metrics.endpoint(endpoint);
        var future = func.apply(endpointMetrics);
        endpointMetrics.callMetrics.forFuture(future);
        return future;
    }

    private void handleException(HttpClientMetrics.Endpoint metrics, HttpResponse<String> response, UUID headerUuid) {
        metrics.incStatus(response.statusCode());
        if (!HttpStatusClass.SUCCESS.contains(response.statusCode())) {
            logger.error("Project response (status {}) ({}):\n{}, body {}",
                    response.statusCode(), headerUuid, response, response.body());
            if (HttpStatusClass.CLIENT_ERROR.contains(response.statusCode()) && response.statusCode() != HttpStatus.TOO_MANY_REQUESTS.value()) {
                // wouldnt retry
                throw new ClientError(response.body());
            }
            //retry errors
            throw new IllegalStateException("Project response status " + response.statusCode());
        }
    }

    private <T> CompletableFuture<T> executeWithRetries(Class<T> clzResult, Object requestDto, String projectId, String endpoint) {
        CompletableFuture<T> resultFuture = new CompletableFuture<>();
        try {
            runWithRetries(() -> request(clzResult, requestDto, resultFuture, projectId, endpoint), RETRY_CONFIG);
        } catch (Throwable t) {
            resultFuture.completeExceptionally(t);
        }
        return resultFuture;
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    private static class ClientError extends RuntimeException {
        public ClientError(String message) {
            super(message);
        }
    }
}
